/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****

 taKeys.c  key pad module program file


*/
#include "taGlobal.h"
#include "taKeys.h"
#include "taKeysUser.h"
#include "taStarControl.h"
#include "taZ_StarControl.h"
#include "taFT245R.h"
#include "taDDS.h"

#define XDDR  DDRC
#define XPORT PORTC
#define YDDR0_5  DDRJ
#define YPORT0_5 PORTJ
#define YPIN0_5  PINJ
#define YDDR6  DDRB
#define YPORT6 PORTB
#define YPIN6  PINB

static uint8_t KBkind; 

char kyHexKeypad(uint16_t k);
char ky18Keypad(uint16_t k);
char kyEA320Motherboard(uint16_t k);
char ky18KeypadB(uint16_t k);
uint16_t kyGetKeyCode();
char kyReadChar();


//////////////////////////////////////////////////////////
//  key pad switching
//  using HardwareSetting.hsKeypad - set by Hobcat

// 0  -  4x4 Hex keypad
// 1  -  18 key pad - with Display and Stacks keys
// 2  -  25 button keypad with on 320x240 S1D13700 motherboard
// 3  -  18 key pad - without  Display and Stacks keys (soft key c, m, n)
// 4  -  User defined key pad

char kyReadChar()
{
	uint16_t k;
	char keyvalue = 'X';
	
	k = kyGetKeyCode();	// get "raw" matrix code 0x0011 to 0x5555 
	if (k != 0) {
		scWakeupDisplay();
		switch (KBkind) {				// get corresponding symbol from hardware-specific case statement
			case 0: keyvalue = kyHexKeypad(k);			break;
			case 1: keyvalue = ky18Keypad(k);			break;
			case 2: keyvalue = kyEA320Motherboard(k);	break;   
			case 3: keyvalue = ky18KeypadB(k);			break;
			case 4: keyvalue = kyUserDefined(k);		break;
		}
	}
	if (keyvalue != 'X') {
		if (DDSparams.dpDisplayKeyValue == 1) zscShowSwitchValue(keyvalue);
	}
	return keyvalue;	
}		 


void kyKeyPadInit()
{
	KBkind = HardwareSettings.hsKeypad;
	YDDR0_5 &= (unsigned char)0x81;		//  bit  1     in   key pad Y5 in
										//  bits 2-6   in   key pad Y0-Y4 in
	YPORT0_5 |= (unsigned char)0x7E;	//  bit  1     pullup  set for key pad input  Y5
										//  bits 2-6   pullups set for key pad inputs Y0 - Y4
	YDDR6 &= (unsigned char)0xFE;		//  bit  0     in   key pad Y6 in
	YPORT6 |= (unsigned char)0x01;		//  bit  0     pullup  set for key pad input  Y6
}


char kyGetKeyChar()
{
  char ka = kyReadChar();
  if (ka != 'X')
  {
  	while (ka == kyReadChar())
  	{
  	  _delay_ms(10);
  	}
  }
  return ka;
}


char kyGetKeyCharLong()						// Enter here
{
	uint16_t t = 0;
	uint16_t k;
	uint16_t ka = kyReadChar();				// get keypad symbol
	if ( ( (ka >= '0') && ( ka <= '9') ) || (ka == '#') ) // for #, 0, 2, 3, 4, 5, 6, 7, 8, 9,only
	{
		do
		{
			t += 1;								// for t=0 to 30 
			k = kyReadChar(); 					// get symbol again
			_delay_ms(20);						// if the key is not released or changed while t<=30 stay here
		} while ( (k == ka) && (t <= 30) );
		if (t > 30) {ka += 0x80;}   	// time's up! add 128 to key code for dds command
	}
	else
	{
		if (ka != 'X')
		{
			while (ka == kyReadChar())
			{
				_delay_ms(10);
			}
		}
	}
	return ka;								// return the key value
}		

 
// returns (16 * x+1) + y+1  ie: key codes 0x11 to 0x17 - 0x81 to 0x87
// returns 0 if no key pressed
uint16_t kyGetKeyCode()
{
	uint16_t keycode;
	unsigned char ybits;
	unsigned char ymask;
	unsigned char y = 0;
	unsigned char x = 0;
	uint8_t keyno = 0;
	uint8_t key[2];
	
 	key[0] = 0;
	key[1] = 0;
	if (ymax == 5) ymask = 0xe0;
	else if (ymax == 6) ymask = 0xc0;
	else ymask = 0x80;
	
	for(x = 0; x < xmax; x++)
	{
		YDDR0_5 |= (unsigned char)0x7E;		// Set Y pins to output
		YDDR6 |= (unsigned char)0x01;
		_delay_us(5);						// Port pins are already high so this will help weak pullups
		YDDR0_5 &= (unsigned char)0x81;		// Set Y pins to input with weak pullups
		YDDR6 &= (unsigned char)0xFE;
		usbINTenable(3, 0);					// disable Rx interrupt INT3
		XPORT = 0;   						// all bits low - but selected by DDR
		if (x < 5) XDDR  = (unsigned char)(1 << (x+3)); // 1 (ie a 1 in bit 0) shifted left (x+3) places (x0 - x4)
		else XDDR = (unsigned char)(1 << (x-5)); // 1 (ie a 1 in bit 0) shifted left (x-5) places (x5 - x7)
											// so x = 1 puts a 1 in bit 3 (ie 0x08) and all rest 0
											// so selected bit is low and rest high impedance
		_delay_us(15);
											// this will = 0xFF if no key pressed
		ybits = ymask |
				((YPIN0_5 & (unsigned char)0x7c) >> 2) |
				((YPIN0_5 & (unsigned char)0x02) << 4) | 
				((YPIN6 & (unsigned char)0x01) << 6);
		if (RxIntEnabled == 1) usbINTenable(3, 1); // enable Rx interrupt INT3
		if (ybits != (unsigned char)0xFF)
		{
			for(y = 0; y < ymax; y++)
			{				 												 
				if ((ybits & (unsigned char)(1 << y)) == 0)
				{									// Ensure non zero value for all keys so shift function works.
					key[keyno++] = ((x + 1) << 4) + (y + 1);// eg: for y = 3
															// and YPORT with a bit shifted 5 to the the Y3 position
     			}											// ie: bit5. If  zero then switch pushed
      		}
		}
  	}
	XDDR = 0; // set to Hi Z

	keycode = ((key[1] << 8) + key[0]);
	if (keycode) {
		if (DDSparams.dpDisplayKeyValue == 2) zscShowSwitchRawValue(keycode);
	}
	return keycode;
}


///////////////////////////////////////////////////////////////////////////

//	4 x 4 Hex Keypad
//
//	[j] - square brackets = shift char
//
//	button				|	code returned by kyHexKeypad(k)
//						|
//	11	21	31	41		|	1[a]	2[b]	3[c]	A(Menu)[d]
//						|
//	12	22	32	42		|	4[e]	5[f]	6[g]	B(Esc)[h]
//						|
//	13	23	33	43		|	7[i]	8[j]	9[k]	C(Display)[l]
//						|
//	14	24	34	44		|	*[m]	0[n]	#[o]	D(Shift)

char kyHexKeypad(uint16_t k)
{
	char ch = 'X';
	
	if ( ((k & 0x00FF) == 0x0044) && ((k & 0xFF00) != 0x0000 ) )
		k = ((k & 0x00FF) << 8) | ((k & 0xFF00) >> 8);

	switch (k)
	{ 

		case 0x0044:  ch = '!'; break;	// Shift Key is "invisible"

		case 0x0011:  ch = '1'; break;
		case 0x0021:  ch = '2'; break;
		case 0x0031:  ch = '3'; break;
		case 0x0041:  ch = 'M'; break;
		case 0x0051:  ch = 'f'; break;	// 51 = f
		case 0x0012:  ch = '4'; break;
		case 0x0022:  ch = '5'; break;
		case 0x0032:  ch = '6'; break;
		case 0x0042:  ch = 'E'; break;
		case 0x0052:  ch = 'g'; break;	// 52 = g
		case 0x0013:  ch = '7'; break;
		case 0x0023:  ch = '8'; break;
		case 0x0033:  ch = '9'; break;
		case 0x0043:  ch = 'D'; break;
		case 0x0053:  ch = 'h'; break;	// 53 = h
		case 0x0014:  ch = '*'; break;
		case 0x0024:  ch = '0'; break;
		case 0x0034:  ch = '#'; break;
		case 0x0054:  ch = 'i'; break;	// 54 = i
		case 0x0015:  ch = 'a'; break;	// 15 = a
		case 0x0025:  ch = 'b'; break;	// 25 = b
		case 0x0035:  ch = 'c'; break;	// 35 = c
		case 0x0045:  ch = 'd'; break;	// 45 = d
		case 0x0055:  ch = 'e'; break;	// 55 = e

		case 0x4411:  ch = 'a'; break;	// Shift and 1
		case 0x4421:  ch = 'b'; break;	// 2
		case 0x4431:  ch = 'c'; break;	// 3
		case 0x4441:  ch = 'd'; break;	// A
		case 0x4412:  ch = 'e'; break;	// 4
		case 0x4422:  ch = 'f'; break;	// 5
		case 0x4432:  ch = 'g'; break;	// 6
		case 0x4442:  ch = 'h'; break;	// B
		case 0x4413:  ch = 'i'; break;	// 7
		case 0x4423:  ch = 'j'; break;	// 8
		case 0x4433:  ch = 'k'; break;	// 9
		case 0x4443:  ch = 'l'; break;	// C
		case 0x4414:  ch = 'm'; break;	// *
		case 0x4424:  ch = 'n'; break;	// 0
		case 0x4434:  ch = 'o'; break;	// #

	}

	if (ch != 'X') {
		if (ch == '!') {
			if (ShiftKeyFlag) ShiftTimer = 5;
			ch = 'X';
		} else {
			ShiftTimer = 0;
			ShiftKeyFlag = 0;
		}
	}

	return ch;
}

/////////////////////////////////////////////////////////////////////////

//	18 key TrxAVR_Picastar keypad
//
//	[j] - square brackets = shift char
//
//	button			|	code returned by ky18Keypad(k)
//					|
//	11	21	31		|	1[a]		2[b]		3[c]			z (= 41)
//					|
//	12	22	32		|	4[d]		5[e]		6[f]			y (= 42)
//					|
//	13	23	33		|	7[g]		8[h]		9[i]			x (= 43)
//					|
//	14	24	34		|	*[j]		0[k]		#[l]			w (= 44)
//					|
//					|											v (= 52)
//					|
//	15	25	35		|	M(enu)		E(sc)		D(isplay)		u (= 53)
//					|
//	43	45	55		|	c			S(tacks)	Shift			t (= 54)
//
//																s (= 51)
                         
char ky18Keypad(uint16_t k)
{
 	char ch = 'X';

	if ( ((k & 0x00FF) == 0x0055) && ((k & 0xFF00) != 0x0000 ) )
		k = ((k & 0x00FF) << 8) | ((k & 0xFF00) >> 8);

	switch (k)
	{

		case 0x0055:  ch = '!'; break;	// shift key invisible

		case 0x0011:  ch = '1'; break;
		case 0x0021:  ch = '2'; break;
		case 0x0031:  ch = '3'; break;
		case 0x0041:  ch = 'z'; break;	// 41 = z
		case 0x0051:  ch = 's'; break;	// 51 = s
		case 0x0012:  ch = '4'; break;
		case 0x0022:  ch = '5'; break;
		case 0x0032:  ch = '6'; break;
		case 0x0042:  ch = 'y'; break;	// 42 = y
		case 0x0052:  ch = 'v'; break;	// 52 = v
		case 0x0013:  ch = '7'; break;
		case 0x0023:  ch = '8'; break;
		case 0x0033:  ch = '9'; break;
		case 0x0053:  ch = 'u'; break;	// 53 = u
		case 0x0014:  ch = '*'; break;
		case 0x0024:  ch = '0'; break;
		case 0x0034:  ch = '#'; break;
		case 0x0044:  ch = 'w'; break;	// 44 = w
		case 0x0054:  ch = 't'; break;	// 54 = t
		case 0x0015:  ch = 'M'; break;
		case 0x0025:  ch = 'E'; break;
		case 0x0035:  ch = 'D'; break;
		case 0x0043:  ch = 'x'; break;	// 43 = x
		case 0x0045:  ch = 'S'; break;

		case 0x5511:  ch = 'a'; break;	// Shift and 1
		case 0x5521:  ch = 'b'; break;	// 2
		case 0x5531:  ch = 'c'; break;	// 3  
		case 0x5512:  ch = 'd'; break;	// 4 
		case 0x5522:  ch = 'e'; break;	// 5
		case 0x5532:  ch = 'f'; break;	// 6
		case 0x5513:  ch = 'g'; break;	// 7
		case 0x5523:  ch = 'h'; break;	// 8
		case 0x5533:  ch = 'i'; break;	// 9
		case 0x5514:  ch = 'j'; break;	// *
		case 0x5524:  ch = 'k'; break;	// 0
		case 0x5534:  ch = 'l'; break;	// #
	}

	if (ch != 'X') {
		if (ch == '!') {
			if (ShiftKeyFlag) ShiftTimer = 5;
			ch = 'X';
		} else {
			ShiftTimer = 0;
			ShiftKeyFlag = 0;
		}
	}

	return ch;
}

/////////////////////////////////////////////////////////////////////////

//	320x240 EA W320  graphics motherboard
//
//	[j] - square brackets = shift char
//
//	button					|	code returned by kyEA320Motherboard(k)
//							|
//	11	21	31		41		|	1[i]		2[j]		3[k]		z
//							|
//	12	22	32		42		|	4[l]		5[m]		6[n]		y
//							|
//	13	23	33		43		|	7[o]		8			9			x
//							|
//	14	24	34		44		|	*			0			#			w
//							|
//					52		|										v          
//							|
//	15	25	35		53		|	M(enu)		E(sc)		D(isplay)	u
//							|
//	XX	45	55		54		|				S(tacks)	Shift		t
//							|
//					51		|										s
//							|
//							|
//	na na na na na na na	|	l	m	n	o	p	q	r		<<< touch pad

char kyEA320Motherboard(uint16_t k)
{
	char ch = 'X';

	if ( ((k & 0x00FF) == 0x0055) && ((k & 0xFF00) != 0x0000 ) )
		k = ((k & 0x00FF) << 8) | ((k & 0xFF00) >> 8);

	switch (k)
	{ 

		case 0x0055:  ch = '!'; break;  // shift key invisible

		case 0x0011:  ch = '1'; break;
		case 0x0021:  ch = '2'; break;
		case 0x0031:  ch = '3'; break;
		case 0x0041:  ch = 'z'; break;
		case 0x0051:  ch = 's'; break;
		case 0x0012:  ch = '4'; break;
		case 0x0022:  ch = '5'; break;
		case 0x0032:  ch = '6'; break;
		case 0x0042:  ch = 'y'; break;
		case 0x0052:  ch = 'v'; break;
		case 0x0013:  ch = '7'; break;
		case 0x0023:  ch = '8'; break;
		case 0x0033:  ch = '9'; break;
		case 0x0043:  ch = 'x'; break;
		case 0x0053:  ch = 'u'; break;
		case 0x0014:  ch = '*'; break;
		case 0x0024:  ch = '0'; break;
		case 0x0034:  ch = '#'; break;
		case 0x0044:  ch = 'w'; break;
		case 0x0054:  ch = 't'; break;
		case 0x0015:  ch = 'M'; break;
		case 0x0025:  ch = 'E'; break;
		case 0x0035:  ch = 'D'; break;
		case 0x0045:  ch = 'S'; break;

		case 0x5511:  ch = 'i'; break;	// Shift and 1
		case 0x5521:  ch = 'j'; break;	// 2
		case 0x5531:  ch = 'k'; break;	// 3  
		case 0x5512:  ch = 'l'; break;	// 4 
		case 0x5522:  ch = 'm'; break;	// 5
		case 0x5532:  ch = 'n'; break;	// 6
		case 0x5513:  ch = 'o'; break;	// 7
	}

	if (ch != 'X') {
		if (ch == '!') {
			if (ShiftKeyFlag) ShiftTimer = 5;
			ch = 'X';
		} else {
			ShiftTimer = 0;
			ShiftKeyFlag = 0;
		}
	}

	return ch;
}

////////////////////////////////////////////////////////////////////////

//	18 key TrxAVR_Picastar keypad - No display and no Stacks key
//
//	[j] - square brackets = shift char
//
//	button			|	code returned by ky18KeypadB(k)
//					|
//	11	21	31		|	1[a]		2[b]		3[c]
//					|
//	12	22	32		|	4[d]		5[e]		6[f]
//					|
//	13	23	33		|	7[g]		8[h]		9[i]
//					|
//	14	24	34		|	*[j]		0[k]		#[l]
//					|
//					|
//					|
//	15	25	35		|	M(enu)		E(sc)		Shift
//					|
//	43	45	55		|	c			m			n
//
//
                         
char ky18KeypadB(uint16_t k)
{
	char ch = 'X';

	if ( ((k & 0x00FF) == 0x0035) && ((k & 0xFF00) != 0x0000 ) )
		k = ((k & 0x00FF) << 8) | ((k & 0xFF00) >> 8);

	switch (k)
	{

		case 0x0035:  ch = '!'; break;	// shift key invisible

		case 0x0011:  ch = '1'; break;
		case 0x0021:  ch = '2'; break;
		case 0x0031:  ch = '3'; break;
		case 0x0041:  ch = 'z'; break;	// 41 = z
		case 0x0051:  ch = 's'; break;	// 51 = s
		case 0x0012:  ch = '4'; break;
		case 0x0022:  ch = '5'; break;
		case 0x0032:  ch = '6'; break;
		case 0x0042:  ch = 'y'; break;	// 42 = y
		case 0x0052:  ch = 'v'; break;	// 52 = v
		case 0x0013:  ch = '7'; break;
		case 0x0023:  ch = '8'; break;
		case 0x0033:  ch = '9'; break;
		case 0x0053:  ch = 'u'; break;	// 53 = u
		case 0x0014:  ch = '*'; break;
		case 0x0024:  ch = '0'; break;
		case 0x0034:  ch = '#'; break;
		case 0x0044:  ch = 'w'; break;	// 44 = w
		case 0x0054:  ch = 't'; break;	// 54 = t
		case 0x0015:  ch = 'M'; break;
		case 0x0025:  ch = 'E'; break;
		case 0x0043:  ch = 'c'; break;	// 43 = c
		case 0x0045:  ch = 'm'; break;	// 45 = m
		case 0x0055:  ch = 'n'; break;	// 55 = n

		case 0x3511:  ch = 'a'; break;	// Shift and 1
		case 0x3521:  ch = 'b'; break;	// 2
		case 0x3531:  ch = 'c'; break;	// 3
		case 0x3512:  ch = 'd'; break;	// 4
		case 0x3522:  ch = 'e'; break;	// 5
		case 0x3532:  ch = 'f'; break;	// 6
		case 0x3513:  ch = 'g'; break;	// 7
		case 0x3523:  ch = 'h'; break;	// 8
		case 0x3533:  ch = 'i'; break;	// 9
		case 0x3514:  ch = 'j'; break;	// *
		case 0x3524:  ch = 'k'; break;	// 0
		case 0x3534:  ch = 'l'; break;	// #
	}

	if (ch != 'X') {
		if (ch == '!') {
			if (ShiftKeyFlag) ShiftTimer = 5;
			ch = 'X';
		} else {
			ShiftTimer = 0;
			ShiftKeyFlag = 0;
		}
	}

	return ch;
}
