/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****


 taGraphicsDriverEA320.c    program file
 
Graphics display driver   320x240   
Epson S1D13700 driver chip
 
*/

#include "taGlobal.h"
#include "taGraphicsDriverEA320.h"
#include "taFT245R.h"
#include "taFont16.h"
#include "taFont8.h"
#include "taFontFreqA.h"
#include "taFontFreqAx.h"
#include "taFontFreqB.h"
#include "taFontFreqBx.h"


#define clWhite 1
#define clBlack 0
#define pmCopy 0
#define pmXOR  1

#define ddrGeData  DDRC
#define portGeData PORTC
#define pinGeData  PINC
#define portGeRES  PORTL
#define portGeCS   PORTL
#define portGeRD   PORTG  //  used with generic 8080  only
#define portGeE    PORTG  //  used with 6800 only
#define portGeWR   PORTG  //  used with generic 8080  only
#define portGeRW   PORTG  //  used with 6800 only
#define portGeA0   PORTD

#define GeRES  6
#define GeCS   7
#define GeRD   1  //  used with generic 8080  only
#define GeE    1  //  used with 6800 only
#define GeWR   0  //  used with generic 8080  only
#define GeRW   0  //  used with 6800 only
#define GeA0   7

#define LayerTR 1   // all layers graphic , all text uses graphics
#define LayerRx 2
#define LayerTx 3

#define SAD1 0x0000
#define SAD2 9600
#define SAD3 19200

#define GraphicsRAMsize 9600
#define CGRAMAddress 0xF000

// S1D13700 commands...   
#define  cmdSystem  0x40          // general system settings
#define  cmdSleep  0x53           // enter into standy mode
#define  cmdDisplayOff  0x58      // turn the display off
#define  cmdDisplayOn  0x59       // turn the display on
#define  cmdScroll  0x44          // setup text and graphics address regions
#define  cmdCsrForm  0x5D         // set cursor size
#define  cmdCsrDirRight  0x4C     // cursor moves right after write to display memory
#define  cmdCsrDirLeft  0x4D      // cursor moves left after write to display memory
#define  cmdCsrDirUp  0x4E        // cursor moves up after write to display memory
#define  cmdCsrDirDown  0x4F      // cursor moves down after write to display memory
#define  cmdCGRAMAddress  0x5C    // configure character generator RAM address
#define  cmdHDotScroll  0x5A      // set horizontal scroll rate
#define  cmdOverlay  0x5B         // configure how layers overlay
#define  cmdSetCsrAddress  0x46   // set the cursor address
#define  cmdGetCsrAddress  0x47   // read the cursor address
#define  cmdDisplayWrite  0x42    // write to display memory
#define  cmdDisplayRead  0x43     // read from display memory

// frequency display - each digit 3 bytes wide on screen 
// keep it simple (and fast) - align with screen bytes
#define geFreqATopLeftPixelX 64        // pixel position  - should be at start of byte
#define geFreqATopLeftPixelY 2        //  these chars have three clear pixel riws at top 
#define geFreqBTopLeftPixelX 132        // pixel position  - should be at start of byte
#define geFreqBTopLeftPixelY 40        //  these chars have three clear pixel riws at top 
#define geLetterSpacing 1 
#define geLineSpacing 18

static const uint16_t GLCDwidth = 320;
static const uint16_t GLCDheight = 240;
static uint8_t GraphicsEnabled;
static uint8_t Layers;
static uint8_t CurrentLayer;
static uint16_t CurrentLayerMemPos;
static uint16_t CurrentLayerRAMsize;
static uint16_t FreqATopLeftPosX;  // VFO A byte position of top left frequency digit - 3 bytes wide
static uint16_t FreqATopLeftPosY;  // VFO A byte position of top left frequency digit - 32 bytes high
static uint16_t FreqBTopLeftPosX;  // VFO B byte position of top left frequency digit - 2 bytes wide
static uint16_t FreqBTopLeftPosY;  // VFO B byte position of top left frequency digit - 24 bytes high
static uint8_t FreqDigit[8];
static uint8_t chlen;
static uint32_t PixLong;
static uint8_t NoofPixBytes;
static uint16_t VerticalWord[16];
static uint32_t HorizontalLong;
static uint32_t MaskA;
static uint32_t MaskB;
static struct TPen Pen;
static struct TPositionS1D13700 Pos;

struct TBrush Brush;
uint8_t InvertPixels;
uint16_t gedNextCharX;

void gedCommand(uint8_t cmd);
uint8_t gedGetData();
void gedCsrMemPosition(uint16_t memstart);
void gedSetPosition();
void gedSetData(uint8_t data);
uint8_t gedApplyLine(uint8_t pLine, uint8_t pMask);
void gedLoadFont16();
void gedWriteDigitA(uint8_t value, uint8_t n, uint8_t ActiveVfo);
void gedWriteDigitB(uint8_t value, uint8_t n, uint8_t ActiveVfo);


void gedShowLayer(uint8_t n, uint8_t on) __attribute__((section(".highmem")));
void gedShowLayer(uint8_t n, uint8_t on)   //external
{
  Layers &= ~(1 << 2*n);
	Layers |=  (on << 2*n);
	gedCommand(cmdDisplayOn);
	gedSetData(Layers);
}
  

void gedFlashLayer(uint8_t n, uint8_t flash) __attribute__((section(".highmem")));
void gedFlashLayer(uint8_t n, uint8_t flash)
{
  Layers &= ~(1 << (2*n+1));
	Layers |=  (flash << (2*n+1));
	gedCommand(cmdDisplayOn);
	gedSetData(Layers);
}
  

void gedSwitchLayer(uint8_t pLayer) __attribute__((section(".highmem")));
void gedSwitchLayer(uint8_t pLayer)    // external
{
  CurrentLayer = pLayer;
	switch(pLayer)
	{
	  case LayerTR:
		{
		  CurrentLayerMemPos = SAD1;
			CurrentLayerRAMsize = GraphicsRAMsize;
			break;
    }
	  case LayerRx: 
		{
		  CurrentLayerMemPos = SAD2;
			CurrentLayerRAMsize = GraphicsRAMsize;
			break;
    }
	  case LayerTx: 
		{
		  CurrentLayerMemPos = SAD3;
			CurrentLayerRAMsize = GraphicsRAMsize;
			break;
    }
  }
}


void gedInitEA320Graphics() __attribute__((section(".highmem")));
void gedInitEA320Graphics()
{
	InvertPixels = 0x00;
	if (zg)
	{
		GraphicsEnabled=1;
	}
	else
	{
		GraphicsEnabled=0;
		return;
	}
	uint16_t i;
	FreqATopLeftPosX = geFreqATopLeftPixelX /8;   
	FreqATopLeftPosY = geFreqATopLeftPixelY;
	FreqBTopLeftPosX = geFreqBTopLeftPixelX /8;   
	FreqBTopLeftPosY = geFreqBTopLeftPixelY;
	Pos.x = 0;
	Pos.y = 0;
	ddrGeData = 0xFF;             // output
	portGeRES |= (1 << GeRES);   // RES = 1  
	portGeCS  |= (1 << GeCS);    // CS  = 1
	portGeA0  |= (1 << GeA0);    // A0 = 1
	if (EA320_6800 == 1)
	{	// 6800 reset
  		portGeRW  |=  (1 << GeRW);    // RW = 1
		portGeE   &= ~(1 << GeE);     // EN = 0
	}
	else
	{  	// generic (8080) reset
		portGeWR |=  (1 << GeWR);    // WR = 1  
		portGeRD |=  (1 << GeRD);    // RD = 1
	}
	portGeRES &= ~(1 << GeRES);   // RES = 0
	_delay_ms(2);
	portGeRES |=  (1 << GeRES);   // RES = 1  
	_delay_ms(5);

	gedCommand(cmdSystem);  // system set command
	gedSetData(0x35);	// for ext RAM 256 chars 16x8  use 0x35  int 5x7 use 0x30
	gedSetData(0x87);	// char width = 8 pixels
	gedSetData(0x0F);	// char is 16 line high
	gedSetData(0x27);	//  40 bytes wide
	gedSetData(0x31);	//  10 blanking bytes
	gedSetData(0xEF);	// L/F = 240 lines
	gedSetData(0x28);	// virtual screen = real screen (40 bytes wide)
	gedSetData(0x00);	//                     "
	gedCommand(cmdScroll);
	gedSetData(SAD1 % 0x100);
	gedSetData(SAD1 / 0x100);
	gedSetData(0xF0);   // SL1 = 240   
	gedSetData(SAD2 % 0x100);
	gedSetData(SAD2 / 0x100);
	gedSetData(0xF0);   // SL2 = 240
	gedSetData(SAD3 % 0x100);
	gedSetData(SAD3 / 0x100);
	gedSetData(0x00);
	gedSetData(0x00);
	gedCommand(cmdCGRAMAddress);
	gedSetData(CGRAMAddress % 0x100);
	gedSetData(CGRAMAddress / 0x100);
	gedCommand(cmdHDotScroll);
	gedSetData(0x00);
	gedCommand(cmdOverlay);
	gedSetData(0x1C);  // 3 graphics layers or'd together
	gedCommand(cmdDisplayOff);
	Layers = 0b00010100; 
	gedSetData(Layers);
	gedCommand(cmdCsrForm);
	gedSetData(0x00);
	gedSetData(0x80);
//	geCommand(cmdCsrDirRight);
	gedCsrMemPosition(0);
	gedCommand(cmdDisplayWrite);
	for (i = 1; i <= 28800; i++)
	{
		gedSetData(0);	//   clear display RAM area
	}
	gedCsrMemPosition(CGRAMAddress);
	for (i = 1; i <= 2048; i++)
	{
		gedSetData(0);	// clear CGRAM area
	}
	gedLoadFont16();
	gedCsrMemPosition(0);
	gedCommand(cmdDisplayOn);
	Layers = 0b00010100;   //   Tx layer is off
	gedSetData(Layers);
	gedSwitchLayer(LayerTR);
	gedCls(0);
	gedSwitchLayer(LayerRx);
	gedCls(0);
	gedSwitchLayer(LayerTx);
	gedCls(0);
	Brush.Color = clWhite;
	gedShowLayer(1,1);
	gedShowLayer(2,1);   // Rx layer on 
	gedShowLayer(3,0);   // Tx layer off
//	portGeWR |=  (1 << GeWR);    // WR = 1  //CMS
//	portGeRD |=  (1 << GeRD);    // RD = 1  //CMS
	gedSwitchLayer(LayerTR);
}


void gedCommand(uint8_t cmd) __attribute__((section(".highmem")));
void gedCommand(uint8_t cmd)
{
	if(GraphicsEnabled==0){return;}
	usbINTenable(3,0); // disable Rx interrupt INT3
	if(EA320_6800==1)
	{
	  	ddrGeData = 0xFF;            // set port to output
	  	portGeData = cmd;            // command byte onto data lines  	
  		portGeA0 |=  (1 << GeA0);    // A0 = 1   command
	  	portGeRW &= ~(1 << GeRW);    // RW = 0
		portGeCS &= ~(1 << GeCS);    // CS = 0
		portGeE  |=  (1 << GeE);     // E = 1
		__asm__ volatile ("nop\r\n nop\r\n nop\r\n nop\r\n" : : );	//wait 4 instructions
		portGeE  &= ~(1 << GeE);     // E = 0
	}
	else
	{
  		ddrGeData = 0xFF;            // set port to output
  		portGeData = cmd;            // command byte onto data lines  
  		portGeA0 |=  (1 << GeA0);    // A0 = 1   command
  		portGeRD |=  (1 << GeRD);    // RD = 1   high for writing mode
		portGeCS &= ~(1 << GeCS);    // CS = 0
		portGeWR &= ~(1 << GeWR);    // WR = 0    write to display
		__asm__ volatile ("nop\r\n nop\r\n nop\r\n nop\r\n" : : );	//wait 4 instructions
		portGeWR |=  (1 << GeWR);    // WR = 1
	}
	portGeCS |=  (1 << GeCS);    // CS = 1
	ddrGeData = 0x00;            // set port to input
  	portGeData = 0x00;           // data byte to 0 - Hi Z
	if (RxIntEnabled == 1) usbINTenable(3,1); // enable Rx interrupt INT3
}


void gedSetData(uint8_t data) __attribute__((section(".highmem")));
void gedSetData(uint8_t data)
{
	if(GraphicsEnabled==0){return;}
	usbINTenable(3,0); // disable Rx interrupt INT3
	if(EA320_6800==1)
	{
		ddrGeData = 0xFF;            // set port to output
		portGeData = data;           // data byte onto data lines
		portGeA0 &= ~(1 << GeA0);    // A0 = 0   display data and parameter write
		portGeRW &= ~(1 << GeRW);    // RW = 0  write mode
		portGeCS &= ~(1 << GeCS);    // CS = 0
		portGeE  |=  (1 << GeE);     // E = 1    write to display
		__asm__ volatile ("nop\r\n nop\r\n nop\r\n nop\r\n" : : );	//wait 4 instructions
		portGeE  &= ~(1 << GeE);     // E = 0
	}
	else
	{
  		ddrGeData = 0xFF;            // set port to output
	  	portGeData = data;           // data byte onto data lines
		portGeA0 &= ~(1 << GeA0);    // A0 = 0   display data and parameter write
  		portGeRD |=  (1 << GeRD);    // RD = 1   high for writing mode
		portGeCS &= ~(1 << GeCS);    // CS = 0
		portGeWR &= ~(1 << GeWR);    // WR = 0    write to display
		__asm__ volatile ("nop\r\n nop\r\n nop\r\n nop\r\n" : : );	//wait 4 instructions
		portGeWR |=  (1 << GeWR);    // WR = 1
	}
	portGeCS |=  (1 << GeCS);    // CS = 1
	ddrGeData = 0x00;            // set port to input
  	portGeData = 0x00;           // data byte to 0 - Hi Z
	if (RxIntEnabled == 1) usbINTenable(3,1); // enable Rx interrupt INT3
}


uint8_t gedGetData() __attribute__((section(".highmem")));
uint8_t gedGetData()
{
  	uint8_t data;
  	
	if(GraphicsEnabled==0){return 0;}
	usbINTenable(3,0); // disable Rx interrupt INT3
	if(EA320_6800==1)
	{
		portGeA0 |=  (1 << GeA0);    // A0 = 1   display data and cursor address read
  		portGeRW |=  (1 << GeRW);    // RW = high for reading
		portGeCS &= ~(1 << GeCS);    // CS = 0
  		ddrGeData = 0x00;            // set port to input
	  	portGeData = 0x00;           // data byte to 0 - Hi Z
		portGeE  |=  (1 << GeE);     // E = 1  read 
		//	4 * 3 * 62.5ns delay
		__asm__ volatile (
			"ldi r24,4 \r\n"
			"1: dec r24 \r\n"
			"brne 1b \r\n"
			: : : "r24"
		);
		data = pinGeData;            // read data from bus
		portGeE  &= ~(1 << GeE);     // E = 0
	}
  else
	{
		portGeA0 |=  (1 << GeA0);    // A0 = 1   display data and cursor address read
  		portGeWR |=  (1 << GeWR);    // WR = high for reading
		portGeCS &= ~(1 << GeCS);    // CS = 0
		ddrGeData = 0x00;            // set port to input
	  	portGeData = 0x00;           // data byte to 0 - Hi Z
		portGeRD &= ~(1 << GeRD);    // RD = 0  read low starts sequence
		//	4 * 3 * 62.5ns delay
		__asm__ volatile (
			"ldi r24,4 \r\n"
			"1: dec r24 \r\n"
			"brne 1b \r\n"
			: : : "r24"
		);
		data = pinGeData;            // read data from bus
		portGeRD |=  (1 << GeRD);    // RD = 1
	}
	portGeCS |=  (1 << GeCS);    // CS = 1
	// ddrGeData = 0xFF;            // set port to output
	if (RxIntEnabled == 1) usbINTenable(3,1); // enable Rx interrupt INT3
	return data;
}


void gedCsrMemPosition(uint16_t pMemStart) __attribute__((section(".highmem")));
void gedCsrMemPosition(uint16_t pMemStart)
{
	gedCommand(cmdSetCsrAddress);
	gedSetData(pMemStart & 0x00FF); // low byte
	gedSetData(pMemStart >> 8);	// high byte 
}


void gedSetPosition() __attribute__((section(".highmem")));
void gedSetPosition()
{
	uint16_t MemPos;
	
	MemPos = CurrentLayerMemPos + Pos.x + 40*Pos.y;  // 40 bytes accross 320 pixel screen
	gedCsrMemPosition(MemPos);
}


void gedWriteByte(uint8_t pValue) __attribute__((section(".highmem")));
void gedWriteByte(uint8_t pValue)
{
	gedSetPosition();
	gedCommand(cmdDisplayWrite);
	gedSetData(pValue^InvertPixels);
}


uint8_t gedReadByte() __attribute__((section(".highmem")));
uint8_t gedReadByte()
{
	gedSetPosition();
	gedCommand(cmdDisplayRead);
	return gedGetData()^InvertPixels;
}


void gedSetPixel(uint16_t pX, uint16_t pY) __attribute__((section(".highmem")));
void gedSetPixel(uint16_t pX, uint16_t pY)
{
	uint16_t LastPosX;
	uint8_t XBit;
	uint8_t Pixel;

	if ((pX < GLCDwidth) && (pY < GLCDheight))
	{
		LastPosX = Pos.x;
		XBit = 0b10000000 >> (pX % 8);
		Pos.y = pY;
		Pos.x = pX / 8;
		Pixel = gedReadByte();
    
		if (Pen.Color == clWhite)    // white pen
		{
			if(Pen.Mode == pmCopy)
			{
				XBit = ~XBit;
				Pixel &= XBit;
			}
		}
		else
		{
			if(Pen.Mode != pmXOR) {Pixel |= XBit;}
			else
			{
				if ((Pixel & XBit) == 0) {Pixel |= XBit;}
				else
				{
					XBit = ~XBit;
					Pixel &= XBit;
				}
			}
		}
		gedWriteByte(Pixel);
		Pos.x = LastPosX;  
	}
}


void gedCls(uint8_t pColor) __attribute__((section(".highmem")));
void gedCls(uint8_t pColor)
{
	uint16_t MemPos;
	uint8_t fill;
	if(pColor==1){fill=0xFF;}else{fill=0x00;};
	gedCsrMemPosition(CurrentLayerMemPos);
	gedCommand(cmdDisplayWrite);
  MemPos = 0;
	do
	{
	  gedSetData(fill^InvertPixels);
		MemPos += 1;
	}
	while(MemPos<CurrentLayerRAMsize);
	gedCsrMemPosition(CurrentLayerMemPos);	
	gedCommand(cmdDisplayOn);
	gedSetData(Layers);
}


uint8_t gedApplyLine(uint8_t pLine, uint8_t pMask) __attribute__((section(".highmem")));
uint8_t gedApplyLine(uint8_t pLine, uint8_t pMask)
{
	if (Brush.Color == clBlack) return (pLine & (~pMask));
	else return (pLine | pMask);
}


void gedFillRect(uint16_t pX1, uint16_t pY1, uint16_t pX2, uint16_t pY2) __attribute__((section(".highmem")));
void gedFillRect(uint16_t pX1, uint16_t pY1, uint16_t pX2, uint16_t pY2)
{
	uint16_t FirstX, LastX;
	uint8_t MaskL, MaskR;
	uint8_t Line;
	
	FirstX = pX1 >> 3;
	LastX = pX2 >> 3;
	MaskL = 0xFF >> (pX1 & 0x0007);
	MaskR = 0xFF << (7 - (pX2 & 0x0007));
	Pos.x = FirstX;
	Pos.y = pY1;
	
	do
	{
		Line = gedReadByte();
		if (Pos.x == LastX) gedWriteByte(gedApplyLine(Line, (MaskL & MaskR)));
		else
		{
			gedWriteByte(gedApplyLine(Line, MaskL));
			Pos.x += 1;
			if (Pos.x < LastX)
			{
				gedCommand(cmdDisplayWrite);
				do
				{
					if(Brush.Color == clBlack) gedSetData(0x00 ^ InvertPixels);
					else gedSetData(0xFF ^ InvertPixels);
					Pos.x += 1;
				} while (Pos.x < LastX);
			}
			Line = gedReadByte();
			gedWriteByte(gedApplyLine(Line, MaskR));
			Pos.x = FirstX;
		}
		Pos.y += 1;
	} while (Pos.y <= pY2);
}


void gedRect(uint16_t pX1, uint16_t pY1, uint16_t pX2, uint16_t pY2,uint8_t width) __attribute__((section(".highmem")));
void gedRect(uint16_t pX1, uint16_t pY1, uint16_t pX2, uint16_t pY2, uint8_t width)
{
	gedHorizLine(pX1,pX2,pY1,width);
	gedHorizLine(pX1,pX2,pY2,width);
	gedVertLine(pY1,pY2,pX1,width);
	gedVertLine(pY1,pY2,pX2,width);
}

			 
// horizontal line   - pY is top edge of line
void gedHorizLine(uint16_t pX1, uint16_t pX2, uint8_t pY, uint8_t width) __attribute__((section(".highmem")));
void gedHorizLine(uint16_t pX1, uint16_t pX2, uint8_t pY, uint8_t width)
{
  gedFillRect(pX1,pY,pX2,(pY+width-1));
}


// vertical line   - pX is left edge of line
void gedVertLine(uint8_t pY1, uint8_t pY2, uint16_t pX, uint8_t width) __attribute__((section(".highmem")));
void gedVertLine(uint8_t pY1, uint8_t pY2, uint16_t pX, uint8_t width)
{
  gedFillRect(pX,pY1,(pX+width-1),pY2);
}




////////////////////////////////////////////////////////////////
////   TEXT HANDLING ///////





void gedLoadFont16() __attribute__((section(".highmem")));
void gedLoadFont16()
{
  uint8_t membyte;
	uint8_t ascii;
	uint8_t width;
	uint8_t col;
	uint8_t row;
	uint16_t* pcol;
  uint16_t addr;
	gedCsrMemPosition(CGRAMAddress);
	gedCommand(cmdDisplayWrite);
	for(addr=0;addr<32*16;addr++){gedSetData(0);};  //zero ascii 0 - 31
	for(ascii=32;ascii<=127;ascii++)
	{
	  width = pgm_read_word(&lentbl_L[ascii-32]);  //  font width   ie no of columns
		pcol = (uint16_t*)pgm_read_word(&chrtbl_L[ascii-32]);
    gedCsrMemPosition(CGRAMAddress + 16*ascii);
	  gedCommand(cmdDisplayWrite);
		for(row=0;row<=15;row++)
		{
		  membyte = 0;
			for(col=0;col<width;col++)
			{
				if((pgm_read_word(&pcol[col]) & (1 << row))!=0){membyte |= (1 << (7-col));};
			}
			gedSetData(membyte^InvertPixels);
		}	  
  }


}





void gedPosWriteChar(uint8_t pX, uint8_t pY, char ch) __attribute__((section(".highmem")));
void gedPosWriteChar(uint8_t pX, uint8_t pY, char ch)
{  
  Pos.x = pX;
	Pos.y = pY;
	gedWriteByte(ch);
}


void gedSetCharPos(uint8_t pX, uint8_t py) __attribute__((section(".highmem")));
void gedSetCharPos(uint8_t pX, uint8_t py)
{
  gedCsrMemPosition(0);

}





/////////////////////////////////////////////////////////////////
/////   Large Font for frequency display /

void gedWriteDigit(uint8_t value, uint8_t n, uint8_t Vfo, uint8_t ActiveVfo) __attribute__((section(".highmem")));
void gedWriteDigit(uint8_t value, uint8_t n, uint8_t Vfo, uint8_t ActiveVfo)
{
  if(Vfo==0)
	{gedWriteDigitA(value,n,ActiveVfo);}
	else
	{gedWriteDigitB(value,n,ActiveVfo);}
}


void gedWriteDigitA(uint8_t value, uint8_t n, uint8_t ActiveVfo) __attribute__((section(".highmem")));
void gedWriteDigitA(uint8_t value, uint8_t n, uint8_t ActiveVfo)
{
  uint8_t x;
	uint8_t y;
	uint8_t * pDigitBytes;
	if(ActiveVfo==0)   // ie If A is active
	{pDigitBytes = (uint8_t*)pgm_read_word(&ffaCharBytes[value]);}  // normal digits
	else
	{pDigitBytes = (uint8_t*)pgm_read_word(&ffaxCharBytes[value]);};  // greyed digits
  if((value<12)||(value==255))
	{Pos.x = FreqATopLeftPosX + pgm_read_byte(&ffaDigitPos[n]);}  // same for normal and greyed
	else
	{Pos.x = FreqATopLeftPosX - 3;};  // VFO tag char 24 pixels left of freq digits 
  for(y=0;y<=31;y++)
	{
  	Pos.y = FreqATopLeftPosY + y;  
	  gedSetPosition();
	  gedCommand(cmdDisplayWrite);
	  for(x=0;x<=2;x++)
		{
		  if(value==255)  // blank leading zero
			{gedSetData(0^InvertPixels);}
			else
			{gedSetData(InvertPixels^pgm_read_byte(&pDigitBytes[x + 3*y]));}
    } 
  }
}


void gedWriteDigitB(uint8_t value, uint8_t n, uint8_t ActiveVfo) __attribute__((section(".highmem")));
void gedWriteDigitB(uint8_t value, uint8_t n, uint8_t ActiveVfo)
{
  uint8_t x;
	uint8_t y;
	uint8_t * pDigitBytes;
	if(ActiveVfo==1)   // ie If B is active
	{pDigitBytes = (uint8_t*)pgm_read_word(&ffbCharBytes[value]);}  // normal digits
	else
	{pDigitBytes = (uint8_t*)pgm_read_word(&ffbxCharBytes[value]);};  // greyed digits
	if((value<12)||(value==255))
	{Pos.x = FreqBTopLeftPosX + pgm_read_byte(&ffbDigitPos[n]);}     // same for normal and greyed
  else
	{Pos.x = FreqBTopLeftPosX - 2;};   // VFO tag char 16 pixels left of freq digits 
	for(y=0;y<=23;y++)
	{
  	Pos.y = FreqBTopLeftPosY + y;  
	  gedSetPosition();
	  gedCommand(cmdDisplayWrite);
	  for(x=0;x<=1;x++)
		{
		  if(value==255)  // blank leading zero
			{gedSetData(0^InvertPixels);}
			else
			{gedSetData(InvertPixels^pgm_read_byte(&pDigitBytes[x + 2*y]));}
    } 
  }
}



void gedDisplayDifference(int32_t df, uint8_t Vfo, uint8_t ActiveVfo) __attribute__((section(".highmem")));
void gedDisplayDifference(int32_t df, uint8_t Vfo, uint8_t ActiveVfo)
{
	int i,j,pm;
	if(df>0){pm=10;}else{pm=11;};   // plus and minus codes
	gedSwitchLayer(1);
	gedClearFrequencyDisplay(Vfo,1);	
	if(df==0)
	{gedWriteDigit(0,7,Vfo,ActiveVfo);}
	else
	{
		int32_t dfa = labs(df);
		for(i=0;i<=7;i++)
		{
	 		FreqDigit[7-i] = dfa % 10;
    	dfa /= 10;
		}	 
		j=0;
		while((FreqDigit[j]==0)&&(j<7)){j+=1;};
		j-=1;
		i = 7;
		do
		{
			gedWriteDigit(FreqDigit[i],i,Vfo,ActiveVfo);
			i-=1;
		}
		while(i>j);
		if(j>=0){gedWriteDigit(pm,j,Vfo,ActiveVfo);};
	}
}



void gedDisplayFrequency(uint32_t f, uint8_t Vfo, uint8_t ActiveVfo) __attribute__((section(".highmem")));
void gedDisplayFrequency(uint32_t f, uint8_t Vfo, uint8_t ActiveVfo)
{
	uint8_t i, j, zflag;

	for (i = 0; i <= 7; i++) {
		FreqDigit[7-i] = f % 10;
		f /= 10;
	}
	gedSwitchLayer(1);
	for (j = 0, zflag = 0; j <= 7; j++) {
		if (zflag == 0) {
			if (FreqDigit[j] == 0) {
				gedWriteDigit(255, j, Vfo, ActiveVfo);
				continue;
			} else {
				zflag = 1;
			}
		}
		gedWriteDigit(FreqDigit[j], j, Vfo, ActiveVfo);
	}
}


void gedClearFrequencyDisplay(uint8_t vfo, uint8_t decpoints) __attribute__((section(".highmem")));
void gedClearFrequencyDisplay(uint8_t vfo, uint8_t decpoints)
{
	gedSwitchLayer(1);
	Brush.Color = clBlack;
	if(vfo==0)
	{
		uint16_t X = geFreqATopLeftPixelX;
  	uint16_t Y = geFreqATopLeftPixelY;
		gedFillRect(X,Y,X+207,Y+31);	
	}  // digits 8x3bytes =+ two one byte gaps
	else
	{
		uint16_t X = geFreqBTopLeftPixelX;
  	uint16_t Y = geFreqBTopLeftPixelY;
		gedFillRect(X,Y,X+140,Y+23);
	}  // digits 8x3bytes =+ two one byte gaps
	if(decpoints==1){gedFreqDecPoints(vfo);};
}



void gedFreqDecPoints(uint8_t vfo) __attribute__((section(".highmem")));
void gedFreqDecPoints(uint8_t vfo)
{
	gedSwitchLayer(1);
	uint16_t X;
  uint16_t Y;
	Brush.Color = clWhite;
  if(vfo==0)
	{
		Y = geFreqATopLeftPixelY + 29;
		X = geFreqATopLeftPixelX + 8*(3 + pgm_read_byte(&ffaDigitPos[1]));
		gedFillRect(X,Y,X+2,Y+2);
		X = geFreqATopLeftPixelX + 8*(3 + pgm_read_byte(&ffaDigitPos[4]));
		gedFillRect(X,Y,X+2,Y+2);
	}
	else
	{
		Y = geFreqBTopLeftPixelY + 21;
		X = geFreqBTopLeftPixelX + 8*(2 + pgm_read_byte(&ffbDigitPos[1])) - 2;
		gedFillRect(X,Y,X+2,Y+2);
		X = geFreqBTopLeftPixelX + 8*(2 + pgm_read_byte(&ffbDigitPos[4])) - 2;
		gedFillRect(X,Y,X+2,Y+2);
	}
}

void gedInitFrequencyDisplay() __attribute__((section(".highmem")));
void gedInitFrequencyDisplay()
{
	gedSwitchLayer(1);
	gedFreqDecPoints(0);
	gedFreqDecPoints(1);
}

///////////////////////////////////////////////////////////////
/////  16x8  font as graphics proprtional  ////////////

void gedF16WriteChar(uint8_t ch, uint16_t* ppX, uint16_t* ppY) __attribute__((section(".highmem")));
void gedF16WriteChar(uint8_t ch, uint16_t* ppX, uint16_t* ppY)
{
	uint32_t ShiftLong = 1;
	uint8_t i;
	uint8_t x;
	uint8_t y;
	uint8_t b;
	uint8_t xmod;
	uint16_t * pDigitWords = (uint16_t*)pgm_read_word(&chrtbl_L[ch-32]);
	
	chlen = pgm_read_byte(&lentbl_L[ch-32]);
	NoofPixBytes = 1;
	xmod = *ppX % 8;
	b = 8-xmod;
	if((chlen + geLetterSpacing) > b){NoofPixBytes=2;};
	if(chlen > (b + 8 - geLetterSpacing)){NoofPixBytes=3;};	
	Pos.x = *ppX / 8;
	for(x=0;x<chlen;x++)
	{
		VerticalWord[x] = pgm_read_word(&pDigitWords[x]);
	}
	for(y=0;y<=15;y++)
	{
		HorizontalLong = 0x00000000;
		for(x=0;x<chlen;x++)
		{
			if((VerticalWord[x] & (1 << y)) > 0){HorizontalLong |= (ShiftLong << (31-x-xmod));}
  		}
		Pos.y = *ppY + y;
		gedSetPosition();
		gedCommand(cmdDisplayRead);
		PixLong = 0;
		for(i=0;i<NoofPixBytes;i++)
		{
			PixLong |= gedGetData() ^ InvertPixels;
			PixLong <<= 8;
		}
		if(NoofPixBytes<3){PixLong <<= 8;};
		if(NoofPixBytes<2){PixLong <<= 8;};
		MaskA = 0xFFFFFFFF >> xmod;
		MaskB = 0xFFFFFFFF >> (xmod + chlen + geLetterSpacing);
		MaskA &= ~MaskB;
		PixLong &= ~MaskA;  // necessary to clear underlying pixels to allow text overwrite
//		HorizontalLong &= MaskA;  // ? not needed
		PixLong |= HorizontalLong;
		gedSetPosition();
		gedCommand(cmdDisplayWrite);
		for(i=0;i<NoofPixBytes;i++)
		{
			gedSetData((PixLong >> 8*(3-i))^InvertPixels);
		}
	}
	*ppX += chlen + geLetterSpacing;  // position for next (proportional spacing)
}



void gedF16WriteString(char* s, uint16_t X, uint16_t Y) __attribute__((section(".highmem")));
void gedF16WriteString(char* s, uint16_t X, uint16_t Y)
{
	uint16_t StartX = X;
	uint16_t nc = strlen(s);
	char ch;
	for (uint16_t i=0;i<nc;i++)
	{
    	ch = s[i];
		if (ch != 10)
		{
			if (ch == 13)
			{
				Y += geLineSpacing;
				X = StartX;
    		}
			else gedF16WriteChar(s[i],&X,&Y);
		}
	}
	gedNextCharX = X;
}

uint16_t gedF16StringWidth(char* s) __attribute__((section(".highmem")));
uint16_t gedF16StringWidth(char* s)
{
	uint8_t nc = strlen(s);
  uint16_t width = 0;
	for (uint8_t i=0;i<nc;i++)
	{
    width += pgm_read_byte(&lentbl_L[s[i]-32]);
	}
	return width;
}


				 
///////////////////////////////////////////////////////////////
/////  8x8  font as graphics proprtional  ////////////


// In gedF8WriteChar, an inverted character if written with an extra white line 
// above and below the character  - this give a much better highlight bar

void gedF8WriteChar(uint8_t ch, uint16_t* ppX, uint16_t* ppY) __attribute__((section(".highmem")));
void gedF8WriteChar(uint8_t ch, uint16_t* ppX, uint16_t* ppY)
{
	uint32_t ShiftLong = 1;
	uint8_t i;
	uint8_t x;
	uint8_t y;
	uint8_t b;
	uint8_t xmod;
	uint8_t * pDigitWords = (uint8_t*)pgm_read_word(&chrtbl_S[ch-32]);
	chlen = pgm_read_byte(&lentbl_S[ch-32]);
	NoofPixBytes = 1;
	xmod = *ppX % 8;
	b = 8-xmod;
	if ((chlen + geLetterSpacing) > b) NoofPixBytes=2;
	if (chlen > (b + 8 - geLetterSpacing)) NoofPixBytes=3;
	Pos.x = *ppX / 8;
	for (x = 0; x < chlen; x++)
	{
		VerticalWord[x] = pgm_read_word(&pDigitWords[x]);
	}
	uint8_t ym = 7;  // normal char
	uint8_t yd = 0;  // normal char
	if (InvertPixels == 0xFF) {ym = 9; yd = 1;}  // inverted char
	for (y = 0; y <= ym; y++)
	{
		HorizontalLong = 0x00000000;  
		// If Inverted leave HorizontalLong = 0. 
		// This will invert to a char length row of 1 bits at the char position 
		if ((InvertPixels == 0) || ((y > 0) && (y != 9)))  // leave HorizontalLong =0 for rows 0 and 9
		{
			for (x = 0; x < chlen; x++)
			{
				if ((VerticalWord[x] & (1 << (y - yd))) > 0) HorizontalLong |= (ShiftLong << (31-x-xmod));
    		}
		}
		Pos.y = *ppY + y - yd; // if overted, start on pix hight to write white line
		gedSetPosition();
		gedCommand(cmdDisplayRead);
		PixLong = 0;
		for (i = 0; i < NoofPixBytes; i++)
		{
			PixLong |= (gedGetData()^InvertPixels);
			PixLong <<= 8;
		}
		if (NoofPixBytes < 3) PixLong <<= 8;
		if (NoofPixBytes < 2) PixLong <<= 8;
		MaskA = 0xFFFFFFFF >> xmod;
		MaskB = 0xFFFFFFFF >> (xmod + chlen + geLetterSpacing);
		MaskA &= ~MaskB;
		PixLong &= ~MaskA;  // necessary to clear underlying pixels to allow text overwrite
		PixLong |= HorizontalLong;
		gedSetPosition();
		gedCommand(cmdDisplayWrite);
		for (i = 0; i < NoofPixBytes; i++)
		{
			gedSetData((PixLong >> 8*(3-i)) ^ InvertPixels);
		}
	}
	*ppX += chlen + geLetterSpacing;  // position for next (proportional spacing)
	return;
}




void gedF8WriteString(char* s, uint16_t X, uint16_t Y) __attribute__((section(".highmem")));
void gedF8WriteString(char* s, uint16_t X, uint16_t Y)
{
	uint16_t StartX = X;
	uint16_t nc = strlen(s);
	char ch;
	
	for (uint16_t i = 0; i < nc; i++)
	{
		ch = s[i];
		if (ch != 10)
		{
			if (ch == 13)
			{
				Y += geLineSpacing;
				X = StartX;
    		}
			else gedF8WriteChar(s[i], &X, &Y);
		}
	}
	gedNextCharX = X; 
}


uint16_t gedF8StringWidth(char* s) __attribute__((section(".highmem")));
uint16_t gedF8StringWidth(char* s)
{
	uint8_t nc = strlen(s);
  uint16_t width = 0;
	for (uint8_t i=0;i<nc;i++)
	{
    width += geLetterSpacing + pgm_read_byte(&lentbl_S[s[i]-32]);
	}
	return width;
}

	
///////////////////////////////////////////

void gedF8CentreWriteString(char* s,uint8_t X,uint8_t Y, uint8_t width) __attribute__((section(".highmem")));
void gedF8CentreWriteString(char* s,uint8_t X,uint8_t Y, uint8_t width)
{
	uint8_t a = strlen(s);
  if(a==0){return ;};
	uint8_t i;
	uint8_t w = 0;
	for(i=0;i<a;i++)
	{
	  w += 1 + pgm_read_byte(&lentbl_S[s[i]-32]);
	}
  w -= 1;   // w is now string width in pixels
	uint16_t Xc = X + (width-w)/2;
  gedF8WriteString(s,Xc,Y);
}

/////////////////////////

void ged5x3WriteString(char* s, uint16_t X, uint16_t Y) __attribute__((section(".highmem")));
void ged5x3WriteString(char* s, uint16_t X, uint16_t Y)
{
	uint16_t nc = strlen(s);
	char ch;
	for (uint16_t i = 0; i < nc; i++)
	{
		ch = s[i];
		gedF8WriteChar(f8Map5x3(ch), &X, &Y);		
	}
}

