/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****

 taCAT.c     program file

   CAT  control module


*/


#include "taGlobal.h"
#include "TrxAVR.h"
#include "taCharLCD.h" 
#include "taUsart.h"
#include "taDDS.h"
#include "taZ_DDS.h"
#include "taStarControl.h"
#include "taZ_StarControl.h"
#include "taStarDSP.h"
#include "taDDScommands.h"
#include "taZ_DDScommands.h"
#include "taSWR.h"


//UART baud rate  - external serial - CAT
#define BAUD0 38400
#define MYUBRR_0 (Xtal/16/BAUD0-1)

#define BufRx0Size 200
#define BufTx0Size 80

#define	portMOX PORTL
#define MOX 3


static const char s_0000[] PROGMEM = "0000";


void ctAC();  //  1
void ctAG();  //  2
void ctAI();  //  3
void ctAL();  //  4
void ctAM();  //  5
void ctAN();  //  6
void ctAR();  //  7
void ctAS();  //  8
void ctBC();  //  9
void ctBD();  // 10
void ctBP();  // 11 
void ctBU();  // 12
void ctBY();  // 13
void ctCA();  // 14
void ctCG();  // 15
void ctCH();  // 16
void ctCI();  // 17
void ctCM();  // 18
void ctCN();  // 19
void ctCT();  // 20
void ctDC();  // 21
void ctDN();  // 22
void ctDQ();  // 23
void ctEX();  // 24
void ctFA();  // 25
void ctFB();  // 26
void ctFC();  // 27
void ctFD();  // 28
void ctFR();  // 29
void ctFS();  // 30
void ctFT();  // 31
void ctFW();  // 32
void ctGT();  // 33
void ctID();  // 34
void ctIF();  // 35
void ctIS();  // 36
void ctKS();  // 37
void ctKY();  // 38
void ctLK();  // 39
void ctLM();  // 40
void ctLT();  // 41
void ctMC();  // 42
void ctMD();  // 43
void ctMF();  // 44
void ctMG();  // 45
void ctML();  // 46
void ctMO();  // 47
void ctMR();  // 48
void ctMU();  // 49
void ctMW();  // 50
void ctNB();  // 51
void ctNL();  // 52
void ctNR();  // 53
void ctNT();  // 54
void ctOF();  // 55
void ctOI();  // 56
void ctOS();  // 57
void ctPA();  // 58
void ctPB();  // 59
void ctPC();  // 60
void ctPI();  // 61
void ctPK();  // 62
void ctPL();  // 63
void ctPM();  // 64
void ctPR();  // 65
void ctPS();  // 66
void ctQC();  // 67
void ctQI();  // 68
void ctQR();  // 69
void ctRA();  // 70
void ctRC();  // 71
void ctRD();  // 72
void ctRG();  // 73
void ctRL();  // 74
void ctRM();  // 75
void ctRU();  // 76
void ctRX();  // 77
void ctSA();  // 78
void ctSB();  // 79
void ctSC();  // 80
void ctSD();  // 81
void ctSH();  // 82
void ctSI();  // 83
void ctSL();  // 84
void ctSM();  // 85
void ctSQ();  // 86
void ctSR();  // 87
void ctSS();  // 88
void ctST();  // 89
void ctSU();  // 90
void ctSV();  // 91
void ctTC();  // 92
void ctTD();  // 93
void ctTI();  // 94
void ctTN();  // 95
void ctTO();  // 96
void ctTS();  // 97
void ctTX();  // 98
void ctTY();  // 99
void ctUL();  // 100
void ctUP();  // 101
void ctVD();  // 102
void ctVG();  // 103
void ctVR();  // 104
void ctVX();  // 105
void ctXT();  // 106



const uint8_t ComTable[26][26]  PROGMEM = {
/// A	 B	C	 D	E  F  G  H  I  J  K  L  M  N  O  P  Q  R  S  T  U  V  W  X  Y  Z
			    
	{ 0, 0, 1, 0, 0, 0, 2, 0, 3, 0, 0, 4, 5, 6, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 0}, // A   AC AG AI AL AM AN AR AS     1,2,3,4,5,6,7,8          
	{ 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,11, 0, 0, 0, 0,12, 0, 0, 0,13, 0}, // B   BC BD BP BU BY     .......  9,10,11,12,13    
	{14, 0, 0, 0, 0, 0,15,16,17, 0, 0, 0,18,19, 0, 0, 0, 0, 0,20, 0, 0, 0, 0, 0, 0}, // C   CA CG CH CI CM CN CT        14,15,16,17,18,19,20    
	{ 0, 0,21, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,22, 0, 0,23, 0, 0, 0, 0, 0, 0, 0, 0, 0}, // D   DC DN DQ  ..............    21,22,23    
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,24, 0, 0}, // E   EX .......................  24          
	{25,26,27,28, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,29,30,31, 0, 0,32, 0, 0, 0}, // F   FA FB FC FD FR FS FT FW  .  25,26,27,28,29,30,31,32         
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,33, 0, 0, 0, 0, 0, 0}, // G   GT  .....................   33 
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, // H             
	{ 0, 0, 0,34, 0,35, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,36, 0, 0, 0, 0, 0, 0, 0}, // I   ID IF IS .................. 34,35,36    
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, // J             
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,37, 0, 0, 0, 0, 0,38, 0}, // K   KS KY ..................... 37,38 
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,39, 0,40, 0, 0, 0, 0, 0, 0,41, 0, 0, 0, 0, 0, 0}, // L   LK LM LT .................  39,40,41          
	{ 0, 0,42,43, 0,44,45, 0, 0, 0, 0,46, 0, 0,47, 0, 0,48, 0, 0,49, 0,50, 0, 0, 0}, // M   MC MD MF MG ML MO MR MU MW  42,43,44,45,46,47,48,49,50            
	{ 0,51, 0, 0, 0, 0, 0, 0, 0, 0, 0,52, 0, 0, 0, 0, 0,53, 0,54, 0, 0, 0, 0, 0, 0}, // N   NB NL NR NT  .............. 51,52,53,54    
	{ 0, 0, 0, 0, 0,55, 0, 0,56, 0, 0, 0, 0, 0, 0, 0, 0, 0,57, 0, 0, 0, 0, 0, 0, 0}, // O   OF OI OS ...................55,56,57          
	{58,59,60, 0, 0, 0, 0, 0,61, 0,62,63,64, 0, 0, 0, 0,65,66, 0, 0, 0, 0, 0, 0, 0}, // P   PA PB PC PI PK PL PM PR PS .58,59,60,61,62,63,64,65,66        
	{ 0, 0,67, 0, 0, 0, 0, 0,68, 0, 0, 0, 0, 0, 0, 0, 0,69, 0, 0, 0, 0, 0, 0, 0, 0}, // Q   QC QI QR  ................. 67 68 69	
	{70, 0,71,72, 0, 0,73, 0, 0, 0, 0,74,75, 0, 0, 0, 0, 0, 0, 0,76, 0, 0,77, 0, 0}, // R   RA RC RD RG RL RM RU RX     70,71,72,73,74,75,76,77        
	{78,79,80,81, 0, 0, 0,82,83, 0, 0,84,85, 0, 0, 0,86,87,88,89,90,91, 0, 0, 0, 0}, // S   SA SB SC SD SH SI SL SM SQ SR SS ST SU SV  78,79,80,81,82,83,84,85,86,87,88,89,90,91         
	{ 0, 0,92,93, 0, 0, 0, 0,94, 0, 0, 0, 0,95,96, 0, 0, 0,97, 0, 0, 0, 0,98,99, 0}, // T   TC TD TI TN TO TS TX TY  ...92,93,94,95,96,97,98,99
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,0,100, 0, 0,0,101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, // U   UL UP  .....................100,101
	{ 0, 0,0,102, 0,0,103, 0, 0, 0, 0, 0, 0, 0, 0, 0,0,104, 0, 0, 0, 0,0,105, 0, 0}, // V   VD VG VR VX  ...............102,103,104,105 
	{ 0, 0, 0, 0, 0,0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, // W               
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,0,106, 0, 0, 0, 0, 0, 0}, // X   XT  .........................106
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, // Y             
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}  // Z 
 };	            

const void* const ComPointerList[107] PROGMEM = {
				0,
                ctAC,  //  1
                ctAG,  //  2
                ctAI,  //  3
                ctAL,	 //	4
                ctAM,	 //	5
                ctAN,  //  6
                ctAR,  //  7
                ctAS,  //  8
                ctBC,  //  9
                ctBD,  // 10
                ctBP,  // 11 
                ctBU,  // 12
                ctBY,  // 13
                ctCA,  // 14
                ctCG,  // 15
                ctCH,  // 16
                ctCI,  // 17
                ctCM,  // 18
                ctCN,  // 19
                ctCT,  // 20
                ctDC,	 // 21
                ctDN,  // 22
                ctDQ,  // 23
                ctEX,  // 24
                ctFA,  // 25
                ctFB,  // 26
                ctFC,  // 27
                ctFD,  // 28
                ctFR,  // 29
                ctFS,  // 30
                ctFT,  // 31
                ctFW,  // 32
                ctGT,  // 33
                ctID,  // 34
                ctIF,  // 35
                ctIS,  // 36
                ctKS,	 // 37
                ctKY,  // 38
                ctLK,  // 39
                ctLM,  // 40
                ctLT,  // 41
                ctMC,	 // 42
                ctMD,  // 43
                ctMF,  // 44
                ctMG,  // 45
                ctML,	 // 46
                ctMO,  // 47
                ctMR,  // 48
                ctMU,  // 49
                ctMW,  // 50
                ctNB,  // 51
                ctNL,	 // 52
                ctNR,	 // 53
                ctNT,  // 54
                ctOF,  // 55
                ctOI,  // 56
                ctOS,  // 57
                ctPA,	 // 58
                ctPB,  // 59
                ctPC,	 // 60
                ctPI,  // 61
                ctPK,  // 62
                ctPL,  // 63
                ctPM,  // 64
                ctPR,  // 65
                ctPS,  // 66
                ctQC,  // 67
                ctQI,  // 68
                ctQR,  // 69
                ctRA,	 // 70
                ctRC,  // 71
                ctRD,	 // 72
                ctRG,  // 73
                ctRL,	 // 74
                ctRM,	 // 75
                ctRU,  // 76
                ctRX,  // 77
                ctSA,  // 78
                ctSB,  // 79
                ctSC,  // 80
                ctSD,	 // 81
                ctSH,  // 82
                ctSI,  // 83
                ctSL,  // 84
                ctSM,  // 85
                ctSQ,  // 86
                ctSR,  // 87
                ctSS,  // 88
                ctST,  // 89
                ctSU,  // 90
                ctSV,  // 91
                ctTC,  // 92
                ctTD,  // 93
                ctTI,  // 94
                ctTN,  // 95
                ctTO,  // 96
                ctTS,  // 97
                ctTX,  // 98
                ctTY,  // 99
                ctUL,  // 100
                ctUP,  // 101
                ctVD,	 // 102
                ctVG,	 // 103
                ctVR,  // 104
                ctVX,	 // 105
                ctXT // 106
};


const uint8_t ComLengthList[107][3] PROGMEM = {
           //    Read  Set Answer
								{  0,  0,  0},  //      0,   
								{  0,  3,  3},  // AC   1
								{  1,  4,  4},  // AG   2
								{  0,  1,  1},  // AI   3
								{  0,  3,  3},  // AL		4
								{  0,  1,  1},  // AM		5
								{  0,  1,  1},  // AN   6
								{  1,  3,  3},  // AR   7
								{  3, 15, 15},  // AS   8
								{  0,  1,  1},  // BC   9
								{255,  0,255},  // BD  10
								{  0,  3,  3},  // BP  11
                {255,  0,255},  // BU  12
								{  0,255,  2},  // BY  13
								{  0,  1,  1},  // CA  14
								{  0,  3,  3},  // CG  15
								{255,  1,255},  // CH  16
								{255,  0,255},  // CI  17
								{  0,  1,  1},  // CM  18
								{  0,  2,  2},  // CN  19
								{  0,  1,  1},  // CT  20
								{  0,  2,  2},  // DC	 21
                {255,  2,255},  // DN  22
								{  0,  1,  1},  // DQ  23
								{  7, 23, 23},  // EX  24
                {  0, 11, 11},  // FA  25
                {  0, 11, 11},  // FB  26
								{  0, 11, 11},  // FC  27
								{  0,255,  8},  // FD  28
                {  0,  1,  1},  // FR  29
								{  0,  1,  1},  // FS  30
                {  0,  1,  1},  // FT  31
                {  0,  4,  4},  // FW  32
                {  0,  3,  3},  // GT  33
                {  0,255,  3},  // ID  34
                {  0,255, 35},  // IF  35
								{  0,  5,  5},  // IS  36
								{  0,  3,  3},  // KS	 37
                {  0, 26,  1},  // KY  38
                {  0,  2,  2},  // LK  39
                {  0,  1,  1},  // LM  40
                {  0,  1,  1},  // LT  41
								{  0,  3,  3},  // MC	 42
                {  0,  1,  1},  // MD  43
                {  0,  1,  1},  // MF  44
                {  0,  3,  3},  // MG  45
								{  0,  3,  3},  // ML	 46
                {  0,  1,  1},  // MO  47
                {  4,255, 47},  // MR  48
                {  0, 10, 10},  // MU  49
                {255, 47,255},  // MW  50
                {  0,  1,  1},  // NB  51
								{  0,  3,  3},  // NL	 52
								{  0,  1,  1},  // NR	 53
                {  0,  1,  1},  // NT  54
                {  0,  9,  9},  // OF  55
                {  0,255, 35},  // OI  56
                {  0,  1,  1},  // OS  57  
								{  0,  1,  2},  // PA	 58
                {  0,  1,  1},  // PB  59
                {  0,  3,  3},  // PC  60
                {255,  1,255},  // PI  61
                {255,255, 48},  // PK  62
								{  0,  6,  6},  // PL	 63
                {  0,  1,  1},  // PM  64
                {  0,  1,  1},  // PR  65
                {  0,  1,  1},  // PS  66
                {  0,  3,  3},  // QC  67
                {255,  0,255},  // QI  68
                {  0,  2,  2},  // QR  69
								{  0,  2,  4},  // RA	 70
                {255,  0,255},  // RC  71
								{  0,  5,  1},  // RD	 72
                {  0,  3,  3},  // RG  73
								{  0,  2,  2},  // RL	 74
								{  0,  1,  5},  // RM	 75
                {  0,  5,  1},  // RU  76
                {255,  0,  1},  // RX  77
                {  0,  7, 15},  // SA  78
                {  0,  1,  1},  // SB  79
                {  0,  1,  1},  // SC  80
								{  0,  4,  4},  // SD	 81
                {  0,  2,  2},  // SH  82
                {255,  9,255},  // SI  83
								{  0,  2,  2},  // SL  84
                {  1,255,  5},  // SM  85
                {  1,  4,  4},  // SQ  86
                {255,  1,255},  // SR  87
                {  0,  1,  1},  // SS  88
                {  0,  2,  2},  // ST  89
                {  1, 11, 11},  // SU  90
                {255,  0,255},  // SV  91
                {  0,  2,  2},  // TC  92
                {255,  1,255},  // TD  93
                {  0,255,  3},  // TI  94
                {  0,  2,  2},  // TN  95
                {  0,  1,  1},  // TO  96
                {  0,  1,  1},  // TS  97
                {255,  1,  1},  // TX  98
                {  0,255,  3},  // TY  99
                {255,255,  1},  // UL  100
                {255,  2,255},  // UP  101
                {  0,  4,  4},  // VD  102
                {  0,  3,  3},  // VG  103
                {255,  1,255},  // VR  104
                {  0,  1,  1},  // VX  105
                {  0,  1,  1}   // XT  106
};


const uint16_t SLtoFreq[12] = {10,50,100,200,300,400,500,600,700,800,900,1000};

const uint16_t SHtoFreq[12] = {1400,1600,1800,2000,2200,2400,2600,2800,3000,3400,4000,5000};

/*
uint8_t ctFreqToSL{uint16_t f)
{
	if {f > 75)	r = ((f - 50) / 100) + 2;
	else {
		if (f > 30) r = 1; else r = 0;
	}
	if (f > 11) f = 11;
	return r;
}


uint8_t ctFreqToSH{uint16_t f)
{
	if (f < 1500) return 0;
	if (f < 3200) r = ((f + 200) / 100) - 7;
	else {
		r = 9;
		if (f > 3700) r = 10;
		if (f > 4500) r = 11;
	}
	return r;
}

*/
			 		

char BufRx0[BufRx0Size];
uint8_t FirstRx0;
uint8_t NextRx0;
uint8_t SkipRx0;
uint8_t ComIndex;  // index for CarCmdj[]

char CatCmdR[16];   //  buffer to assemble dcommand
char CatCmdT[40];
char FreqString[12];
uint8_t KwdNc;     // command string character count
uint8_t KwdNget;   // no of params for Get for identifed command
uint8_t KwdNset;   // no of params for Set for identified command
uint8_t KwdNGSmax; // max parameters of get and Set for this command 
uint8_t KwdIndex;  // command number 1- 31
char KwdGetSet;    //  'S' or 'G'



char BufTx0[BufTx0Size];
uint8_t FirstTx0;
uint8_t NextTx0;
uint8_t SkipTx0;

uint8_t ctRx0BufCount();
uint8_t ctRx0Empty();
uint8_t ctRx0Full();
void ctIntEnableRx0(unsigned char tf);
uint8_t ctReadCharRx0(char* pc);
void ctResetRx0();
char ctReadBufCharRx0();

uint8_t ctTx0BufCount();
uint8_t ctTx0Empty();
uint8_t ctTx0Full();
void    ctResetTx0();
uint8_t ctTx0Empty();
uint8_t ctTx0Full();
uint8_t ctTx0BufCount();
void    ctIntEnableTx0(unsigned char tf);
uint8_t ctTransmitCharTx0(char pc);

uint32_t ctParamsToFreq();
void ctBuildFreqString(uint32_t F);
void ctRITXITFreqString();
char ctGetMode();
void ctSetMode(char m);
uint16_t ctGetValueFromChars(uint8_t a,uint8_t n);
void ctSetValueChars(uint8_t a,uint8_t n, uint16_t value);
void ctDummy();

uint8_t ctFreqToSL(uint16_t f);
uint8_t ctFreqToSH(uint16_t f);


/////////////////////////////////////////////////////

void ctCatInit()
{
	portMOX &= ~(1 << MOX);
	TestIndex = 0;
	psrx0 = StringRx0;
	UBRR0H = (unsigned char)(MYUBRR_0>>8);  // set baud rate
	UBRR0L = (unsigned char)MYUBRR_0;
  UCSR0C = 0x06;  //   asynchronous, no parity, 1 stop bit, 8 data bits 
	UCSR0B = 0x18;   /// set bits  3 and 4 to enable transmission and reception
  ctResetRx0();
	ctResetTx0();
}


///////////////////////////////////////////////////////////////////////
/////////////// TRANSMIT CAT

///////////////////   TX2 - transmit to DSP  //////////////////


ISR(USART0_TX_vect)   // interrupt on character trasmitted on USART2 
{
	if (ctTx0Empty() == 1) return;
	char c = BufTx0[FirstTx0++];
	if (FirstTx0 >= BufTx0Size) FirstTx0 = 0; 
	ctTransmitCharTx0(c);
	return;
}


void ctResetTx0()
{
	FirstTx0 = 0;
	NextTx0 = 0;
	ctIntEnableTx0(1); // enable Tx interrupt INT0
	return;
}


uint8_t ctTx0Empty()
{
  if(FirstTx0 == NextTx0){return 1;};
	return 0;
}



uint8_t ctTx0Full()
{
  uint8_t n = NextTx0;
	if(n < FirstTx0 ){n+=BufTx0Size;};
  if((n - FirstTx0) >= (BufTx0Size-1)) return 1; // buffer full
	return 0;   // buffer not full
}


uint8_t ctTx0BufCount()
{
  if(NextTx0>=FirstTx0)
	{return (NextTx0-FirstTx0);}
	else
	{return (NextTx0 + BufTx0Size - FirstTx0);};  
}



void ctIntEnableTx0(unsigned char tf)
{
  if (tf == 0)
	UCSR0B &= ~(1 << 6);  // disable TXCIE0 (bit 6)
	else
	UCSR0B |= (1 << 6);   // enable TXCIE0 (bit 6)
  return;
}


uint8_t ctTransmitCharTx0(char pc)
{
  int timeout = 0;
	while((UCSR0A & (1<<UDRE0))==0)   // check for ready for transmit  (UDRE0 - bit 5)
	{													 // with be set if arrived here from interrupt
		_delay_us(10);
		if(++timeout > 1000){return 0;};
	}
  UDR0 = pc;  //write char
	return 1;
}



uint8_t ctSendCharTx0(char pc)
{
  if(ctTx0Empty()==1) // && (UCSR0A & (1<<UDRE0)))   // bit 5 = UDRE2
	{
	  ctTransmitCharTx0(pc);
		return 1;
  }
	int timeout = 0;
	while(ctTx0Full()==1)
	{
		_delay_us(10);
		if(++timeout > 1000){return 0;};
	}
  BufTx0[NextTx0++] = pc;
	if(NextTx0 >= BufTx0Size){NextTx0 = 0;}; 
  
	return 1;

}


void ctSendAnswer()
{
	uint8_t i;
	strcat(CatCmdT,";");
	uint8_t nc = strlen(CatCmdT);
	for(i=0;i<nc;i++){ctSendCharTx0(CatCmdT[i]);};
}  



///////////////////////////////////////////////////////////////
///////   RECEIVE CAT

void ctResetRx0()
{
	SkipRx0 = 0;
	FirstRx0 = 0;
	NextRx0 = 0;
	KwdNc = 0;
	ctIntEnableRx0(1); // enable Rx interrupt 
	return;
}





ISR(USART0_RX_vect)   // interrupt on character receieved on USART0 
{
	char c = 0;
	ctReadCharRx0(&c);
	SkipRx0 = ctRx0Full();
	if (SkipRx0 > 0) return;
	BufRx0[NextRx0] = c;
	NextRx0 += 1;
	if (NextRx0 >= BufRx0Size) NextRx0 = 0;
	return;
}



uint8_t  ctRx0Empty()
{
  if(FirstRx0 == NextRx0){return 1;};
	return 0;
}



uint8_t ctRx0Full()
{
  uint8_t n = NextRx0;
	if(n < FirstRx0 ){n+=BufRx0Size;};
  if((n - FirstRx0) >= (BufRx0Size-1)) return 1; // buffer full
	return 0;   // buffer not full
}


uint8_t ctRx0BufCount()
{
  if(NextRx0>=FirstRx0)
	{return (NextRx0-FirstRx0);}
	else
	{return (NextRx0 + BufRx0Size - FirstRx0);};  
}




void ctIntEnableRx0(unsigned char tf)
{
  if (tf == 0)
	UCSR0B &= ~(1 << 7);  // disable RXCIE0 (bit 7)
	else
	UCSR0B |= (1 << 7);   // enable RXCIE0 (bit 7)
  return;
}


uint8_t ctReadCharRx0(char* pc)   // used by interupt routine - char from port to buffer
{
  int timeout = 0;
	while( !(UCSR0A & 0x80))
	{
    timeout += 1;
		_delay_us(10);
		if(++timeout > 1000){return 0;};
	}
  *pc = UDR0;
	return 1;
}




char ctReadBufChar()
{
	if(ctRx0Empty()==1){return 0;};
	char ch = BufRx0[FirstRx0++];
	if(FirstRx0>=BufRx0Size){FirstRx0=0;}
	if((ch>='a')&&(ch<='z')){ch &= 0xDF;};  // convert all lower case to upper case
	return ch;
}

///////////////////////////////////////////////////////

void ctDoCat()
{
  uint8_t np;
	void (*ptask)();   //ptask defined here as a pointer to some function

  while(KwdNc < 50)  // this is just a safety valse
	{
		char ch = ctReadBufChar();
		if(ch==0){return;};
		if(KwdNc<2)
		{
   		if(ch==';'){KwdNc=0; return;};
			if((ch <'A')||(ch >'Z')){return;};
	 	 	CatCmdR[KwdNc++] = ch;	   
	 	 	if(KwdNc==2)
			{
				KwdIndex = pgm_read_byte(&ComTable[CatCmdR[0]-0x41][CatCmdR[1]-0x41]);
				if(KwdIndex==0){KwdNc=0; return;};
				CatCmdT[0] = CatCmdR[0];
				CatCmdT[1] = CatCmdR[1];
				CatCmdT[2] = 0;
				KwdNget = pgm_read_byte(&ComLengthList[KwdIndex][0]);
				KwdNset = pgm_read_byte(&ComLengthList[KwdIndex][1]);
				KwdNGSmax = KwdNget;
  	    if(KwdNset > KwdNGSmax){KwdNGSmax = KwdNset;};  // max no of parameter bytes (get or set)  
			}
		}
		else  // two letter command identified
		{
			if(KwdNc > KwdNGSmax+3){KwdNc=0; return;};
	  	CatCmdR[KwdNc++] = ch;	   		
			if(ch==';')
			{
				CatCmdR[KwdNc] = 0;
				np = KwdNc-3;   // no of param bytes
				KwdNc = 0;    // ready for next command
				if((np!=KwdNget)&&(np!=KwdNset)){return;};
			 	if(np==KwdNget){KwdGetSet='G';}else{KwdGetSet='S';};
      	ptask = (void*)pgm_read_word(&ComPointerList[KwdIndex]);  
				ptask();
				return;
    	}
			else
			{
		  	if(((ch<'0')||(ch>'9'))&&(ch!=' '))
				{
					if((ch>='A')&&(ch<='Z'))
					{KwdNc=1; CatCmdR[0]=ch;}   // unexpected 
					else
					{KwdNc=0;}
					return;
				}
			}
		}
	}
	KwdNc = 0;
}	


/////////////////////////////////////////////////////////////////

/////////////   CAT command utilities  ////////////

uint32_t ctParamsToFreq()
{
  uint8_t i;
	uint32_t f = 0;
  for(i=2;i<=12;i++)
	{
		f *= 10;
		f += (CatCmdR[i] - 0x30);
  }
  return f;
}

void ctBuildFreqString(uint32_t F)
{
	uint8_t i ;
	for(i=0;i<11;i++)
	{
		FreqString[10-i] = (F % 10) + 0x30;
		F /= 10;
  }
	FreqString[11] = 0;
}


void ctRITXITFreqString()
{
  uint8_t i;
	uint32_t diff;
  if(VfoSlot[1].stFreq >= VfoSlot[0].stFreq)
	{
	  diff = VfoSlot[1].stFreq - VfoSlot[0].stFreq;
		FreqString[0] = '+';
	}
	else
	{
	  diff = VfoSlot[0].stFreq - VfoSlot[1].stFreq;
		FreqString[0] = '-';
	}
	if(diff>99999){diff=99999;};
	for(i=0;i<5;i++)
	{
		FreqString[5-i] = (diff % 10) + 0x30;
		diff /= 10;
  }
	FreqString[6] = 0;

}


char ctGetMode()
{
  switch(Mode)
	{
	  case 0: return '1'; break;  // LSB
		case 1: return '2'; break;  // USB
		case 2: return '3'; break;  // CWL
    case 3: return '7'; break;  // CWU
  }
	return '0';  // unused
}


void ctSetMode(char m)
{
  switch (m) {
	  case '1': Mode = 0; break;
	  case '2': Mode = 1; break;
	  case '3': Mode = 2; break;
	  case '4': Mode = 0; break;
	  case '5': Mode = 0; break;
	  case '6': Mode = 0; break;
	  case '7': Mode = 3; break; 
	} 
	scChangeMode();
	ddsSaveModeToLast();
	zscShowSwitches();
	zscShowFilter();
}


uint16_t ctGetValueFromChars(uint8_t a,uint8_t n)
{
  uint8_t i;
	uint16_t r;
	r = 0;
	for(i=a;i<n+a;i++)
	{
	  r *= 10;
		r += (CatCmdR[i] - 0x30);
  }
	return r;
}


void ctSetValueChars(uint8_t a,uint8_t n, uint16_t value)
{
  uint8_t i;
	for(i=0;i<n;i++)
	{
		CatCmdT[n+a-1-i] = (value%10) + 0x30;
		value /=10;		
  }
	CatCmdT[n+a] = 0;
}



void ctDummy()
{
  if(KwdGetSet=='G')
	{
	  uint8_t i;
    uint8_t n = pgm_read_byte(&ComLengthList[KwdIndex][2]); // return length
		for(i=0;i<n;i++){CatCmdT[i+2]='0';};
 		CatCmdT[n+2] = 0;
		ctSendAnswer();
	}
}		
	

uint8_t ctFreqToSL(uint16_t f)
{
	uint8_t r;
	if(f>75)
	{r = (f-50)/100 + 2;}
	else
	{
	  if(f>30){r=1;}else{r=0;};
  }
  if(f>11){f=11;};
	return r;
}


uint8_t ctFreqToSH(uint16_t f)
{
	uint8_t r;
	if(f<1500){return 0;};
	if(f<3200)
	{r = (f+100)/200 - 7;}
	else
	{
	  r = 9;
		if(f>3700){r=10;};
		if(f>4500){r=11;}
  }
	return r;
}

////////////// CAT command routines ///////////////////

void ctAC()
{
	ctDummy();
}



void ctAG()
{
	uint16_t gain;
	if(KwdGetSet=='S')
	{
		if(CatCmdR[2]=='0')   // main transceiever
		{
			gain = 49 + (ctGetValueFromChars(3,3)/5);
			if(gain>100){gain=100;};
			dspChangeParamValue(21,gain); 
		} 
	}
	else
	{
		CatCmdT[2]=CatCmdR[2]; //  send P1 back
		CatCmdT[3]=0;
		if(CatCmdT[2]=='0')
		{gain = dspGetParamValue(21);}
		else
		{gain=49;};
		if(gain<49){gain=49;};  
		gain = (gain-49)*5;
		ctSetValueChars(3,3,gain);
		ctSendAnswer();
	}
}


void ctAI()
{
  ctDummy();
}

void ctAL()
{
	ctDummy();
}


void ctAM()
{
  ctDummy();
}


void ctAN()
{
  ctDummy();  
}


void ctAR()
{
  ctDummy();
}


void ctAS()
{
  ctDummy();
}


void ctBC()
{
	uint8_t bc;       // manual notch
	if(KwdGetSet=='S')
	{
		if(CatCmdR[2]=='1'){bc=1;}
		if(CatCmdR[2]=='0'){bc=0;}
		if(SWmanualnotch!=bc)
		{
			SWmanualnotch = bc;
			dspSendCommand(5,bc);   
			zscShowSwitches();
		}
	}
	else
	{
		if(SWmanualnotch==1){CatCmdT[2]='1';}else{CatCmdT[2]='0';}; 
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
}





void ctBD()
{
	if(KwdGetSet=='S'){cmBandDown();}
}


void ctBP()   /// beat canceller = manual notch
{
	uint16_t bp;
	if(KwdGetSet=='S')
	{
		bp = ctGetValueFromChars(2,3);
		if(bp>46){bp=46;};
		dspChangeParamValue(51,bp*5);   
	}
	else
	{
		bp = dspGetParamValue(51);
		ctSetValueChars(2,3,bp/5);
		ctSendAnswer();
	}
}


void ctBU()
{
	if(KwdGetSet=='S'){cmBandUp();}
}


void ctBY()
{
  ctDummy();
}


void ctCA()
{
  ctDummy();
}


void ctCG()  /// carrier gain - used for Tx drive
{
	uint16_t drv;
	if(KwdGetSet=='S')
	{
		drv = ctGetValueFromChars(2,3);
		if(drv>100){drv=100;};
		dspChangeParamValue(81,drv);   
	}
	else
	{
		drv = dspGetParamValue(81);
		ctSetValueChars(2,3,drv);
		ctSendAnswer();
	}
}


void ctCH()
{
  ctDummy();
}


void ctCI()
{
  ctDummy();
}


void ctCM()
{
  ctDummy();
}



void ctCN()
{
  ctDummy();
}



void ctCT() // yellow param
{
  ctDummy();
}


void ctDC()
{
  ctDummy();
}




void ctDN()  // for VFO A here read 'Active VFO' becuase no VFO B option
{
	ctDummy();
}


void ctDQ()   // Green param
{      
  ctDummy();
}


void ctEX()
{
  ctDummy();
}



void ctFA()
{	
	if(KwdGetSet=='S')
	{
		ddsExtSetFreq(0, ctParamsToFreq());
		if(VFO==0){ClearSlotno = 1;};  // if we are changing this active VFO we have moved away from slot
	}
	else
	{
		ctBuildFreqString(VfoSlot[0].stFreq);
		strcat(CatCmdT,FreqString);
		ctSendAnswer();
	}		
}


void ctFB()
{
	if(KwdGetSet=='S')
	{
		ddsExtSetFreq(1, ctParamsToFreq());
		if(VFO==1){ClearSlotno = 1;};  // if we are changing this active VFO we have moved away from slot	
	}
	else
	{
		ctBuildFreqString(VfoSlot[1].stFreq);
		strcat(CatCmdT,FreqString);
		ctSendAnswer();
	}		
}


void ctFC()
{
  ctDummy();
}



void ctFD()
{
//	ctDummy(); return;
	uint8_t i;
	if(KwdGetSet=='G')
	{
  	for(i=0;i<=7;i++)
		{
			if(i%2 == 0){CatCmdT[i+2]=0xFF;}else{CatCmdT[i+2]=0x01;};
		}
		CatCmdT[10]=0;
		ctSendAnswer();
	}
}






/*
void ctFN()
{
	if(KwdGetSet=='S')
	{
		if(CatCmdR[2]=='1'){ddsHobcatVfoB();}else{ddsHobcatVfoA();};
	}
	else
	{
		strcpy(CatCmdT,"FN");
		if(VFO==1){CatCmdT[2]='1';}else{CatCmdT[2]='0';};
		CatCmdT[3]=0;
		ctSendAnswer();
	}
}
*/


void ctFR()
{
	char vch;
	if(KwdGetSet=='S')
	{
	  vch = CatCmdR[2];
		if((vch=='1')&&(Split==0)) VFO=1; else VFO = 0;
		Mode=255;
		ddsApplyVfo(VFO);
		UpdateDDS = 1;
	  zddsShowBothVfoFreqs();
	}
	else
	{
		if(VFO==0){vch='0';}else{vch='1';};
		CatCmdT[2] = vch;
		CatCmdT[3] = 0;
		ctSendAnswer();
	}		
}


void ctFS()
{
  ctDummy();
}




void ctFT()
{
	char vch;
	uint8_t v = VFO;
	if(KwdGetSet=='S')
	{
	}
	else
	{
		if (Split==0)
		{
			if(v==0){vch='0';}else{vch='1';};
    }
		else
		{
			if(v==0){vch='1';}else{vch='0';};
		}
		CatCmdT[2] = vch;
		CatCmdT[3] = 0;
		ctSendAnswer();
	}		
}


void ctFW()   // there is only one filter width command therefore it must 
{							// adjust the currently selected filter
	uint16_t Hz, OldHz;
	if(KwdGetSet=='S')
	{
		if(Mode<2){return;};
		Hz = ctGetValueFromChars(2,4);
		if(Hz>2300){Hz=2300;};
		OldHz = 20*dspGetParamValue(64);
		if(Hz!=OldHz){dspChangeParamValue(64,Hz/20);};   
	}
	else
	{
		Hz = 20*dspGetParamValue(64);
		ctSetValueChars(2,4,Hz);
		ctSendAnswer();
	}
}


void ctGT()
{
  ctDummy();
}


void ctID()
{
	if(KwdGetSet=='G')
	{
		strcpy(CatCmdT,"ID019");
		ctSendAnswer();
	}
}


void ctIF()
{
	uint8_t ms;
	uint8_t v = VFO;
	{
		ctBuildFreqString(VfoSlot[VFO].stFreq);  // kenwood spec says VFOA - but we need current VFO
		strcat(CatCmdT,FreqString);               // P1
    strcat(CatCmdT,"0011");                   // P2    step = 10E3 = 1000 Hz
		ctRITXITFreqString();
		strcat(CatCmdT,FreqString);               // P3   RIT/XIT  +nnnnn or -nnnnn
		if(RIT==1){CatCmdT[23]='1';}else{CatCmdT[23]='0';};   // P4
		if(XIT==1){CatCmdT[24]='1';}else{CatCmdT[24]='0';};   // P5
    ms = MemStackNo + 7;
		if(ms>=10){ms=0;};
		switch(ParamColour)
		{
			case 1: CatCmdT[25] = 'G'; break;
			case 2: CatCmdT[25] = 'Y'; break;
			case 3: CatCmdT[25] = 'R'; break;  // P6 
		}
		CatCmdT[26]= '6';  				// P7
		CatCmdT[27]=ms + 0x30;  	// P7
		if(Transmit==1){CatCmdT[28]='1';}else{CatCmdT[28]='0';};    // P8
    CatCmdT[29] = ctGetMode();                            // P9
		if(v==0){CatCmdT[30]='0';}else{CatCmdT[30]='1';};         // P10				
		CatCmdT[31] = '0';                                      // P11
		if(v==1){CatCmdT[32]='0';}else{CatCmdT[32]='1';};         // P12
    CatCmdT[33] = '0';
		CatCmdT[34] = '0';
    CatCmdT[35] = '0';
		CatCmdT[36] = '0';
    CatCmdT[37] = 0;
		ctSendAnswer();  
	}
}


void ctIS()
{
	uint16_t Hz, OldHz;
	if(KwdGetSet=='S')
	{
		if(Mode<2){return;};
		Hz = ctGetValueFromChars(3,4);
		if(Hz>1000){Hz=1000;};
		if(Hz<400){Hz=400;}
		OldHz = 10*dspGetParamValue(66);
		if(Hz!=OldHz){dspChangeParamValue(66,Hz/10);};   
	}
	else
	{
		Hz = 10*dspGetParamValue(66);
		CatCmdT[2]=' ';
		ctSetValueChars(3,4,Hz);
		ctSendAnswer();
	}
}




void ctKS()
{
	uint16_t ks;
	if(KwdGetSet=='S')
	{
		ks = ctGetValueFromChars(2,3);
		if(ks>60){ks=60;};
		dspChangeParamValue(41,ks);   
	}
	else
	{
		ks = dspGetParamValue(41);
		ctSetValueChars(2,3,ks);
		ctSendAnswer();
	}
}


void ctKY()
{
  ctDummy();
}


void ctLK()
{
  ctDummy();
}


void ctLM()
{
  ctDummy();
}


void ctLT()
{
	uint8_t filter;
	if (KwdGetSet == 'S') {
		if (CatCmdR[2] == '1') filter = 1;
		if (CatCmdR[2] == '0') filter = 0;
		if (SWfilter != filter) {
			SWfilter = filter;  //  added 3.3.2010   IJS
			if (Mode < 2) SSB_SWfilter = SWfilter;
			else CW_SWfilter = 1; // SWfilter;    IJS Star6a9 changed to 1  depth never context
			scSetWideNarrow();   // sets DSP filter to wide or narrow according the value of SWfilter
		}
	} else {
		if (SWfilter == 1) CatCmdT[2] = '1'; else CatCmdT[2] = '0'; 
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
 
}



void ctMC()
{
	ctDummy();
}


void ctMD()
{
	if(KwdGetSet=='S')
	{
	  ctSetMode(CatCmdR[2]);
	}
	else
	{
		CatCmdT[2] = ctGetMode();
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
}


void ctMF()
{
  ctDummy();
}


void ctMG()
{
  ctDummy();
}




void ctML()
{
	uint16_t m;
	if(KwdGetSet=='S')
	{
		m = 10*ctGetValueFromChars(2,3);
		if(Mode<2)
		{dspChangeParamValue(25,m);}
		else
		{dspChangeParamValue(23,m);}
	}
	else
	{
		if(Mode<2)
		{m = dspGetParamValue(25);}
		else
		{m = dspGetParamValue(23);};
		if(m>90){m=90;};
		ctSetValueChars(2,3,m/10);
		ctSendAnswer();
	}
}


void ctMO()
{
  ctDummy();
}



void ctMR()
{
  ctDummy();
}


void ctMU()
{
  ctDummy();
}


void ctMW()
{
  ctDummy();
}



void ctNB()
{
	if(KwdGetSet=='S')
	{
		if(CatCmdR[2]=='1'){SWnoiseblank=1;}else{SWnoiseblank=0;};
		dspSendCommand(3,SWnoiseblank);   
		zscShowSwitches();
	}
	else
	{
		if(SWnoiseblank==1){CatCmdT[2]='1';}else{CatCmdT[2]='0';}; 
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
}


void ctNL()
{
  ctDummy();}


void ctNR()
{
	if(KwdGetSet=='S')
	{
		if(CatCmdR[2]=='1'){SWdenoise=1;}else{SWdenoise=0;};
		dspSendCommand(1,SWdenoise);   
		zscShowSwitches();
	}
	else
	{
		if(SWdenoise==1){CatCmdT[2]='1';}else{CatCmdT[2]='0';}; 
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
}



void ctNT()
{
	if (KwdGetSet == 'S') {
		if (Mode < 2) {
			if (CatCmdR[2] == '1') SWautonotch = 1; else SWautonotch = 0;
			dspSendCommand(4, SWautonotch);
		} else {
			if (CatCmdR[2] == '1') SWqro = 1; else SWqro = 0;
			dspSendCommand(4, SWqro);
		}
		zscShowSwitches();
	} else {
		if (Mode < 2) {
			if (SWautonotch == 1) CatCmdT[2]='1'; else CatCmdT[2]='0';
		} else {
			if (SWqro == 1) CatCmdT[2]='1'; else CatCmdT[2]='0';
		}
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
}


void ctOF()
{
  ctDummy();
}


void ctOI()
{
  ctDummy();
}



void ctOS()
{
	if(KwdGetSet=='S')
	{
		uint8_t PrevSplit = Split;
		if(CatCmdR[2]=='0')
		{Split=0;}
		else
		{
			if(XRIT()!=0){Split=1;};
		}
	  if(Split!=PrevSplit)
		{
			zddsShowBothVfoFreqs();
			zcmShowSplitMode();      
		};   
	}
	else
	{
		if(Split==1){CatCmdT[2]='1';}else{CatCmdT[2]='0';}; 
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
}



void ctPA()
{
	uint8_t ip;
	if(KwdGetSet=='S')
	{
	  if(CatCmdR[2]=='1'){ip=0;}else{ip=1;};
		if(IP3!=ip)
		{
		  IP3 = ip;
			scSetIP3(IP3);						
			ddsSaveIP3ToLast();
			zscDisplayIP3NF();
		}
  }
	else
	{
    ip = (IP3^0x01);
		CatCmdT[2] = ip + 0x30;
		CatCmdT[3] = '0';
		CatCmdT[4]= 0;
		ctSendAnswer();
	}
}


void ctPB()
{
  ctDummy();
}





void ctPC()
{
	uint8_t pwr;
	if(KwdGetSet=='G')
	{
		if(Transmit==0){pwr=0;}else{pwr = FwdCentiWatts/100;}
		ctSetValueChars(2,3,pwr);
		ctSendAnswer();
  }

}


void ctPI()
{
  ctDummy();
}



void ctPK()
{
  ctDummy();
}




void ctPL()
{
	uint16_t pl;
	if(KwdGetSet=='S')
	{
		pl = ctGetValueFromChars(2,3);   
		if(pl>20){pl=20;};
		dspChangeParamValue(91,pl);   
	}
	else
	{
		pl = dspGetParamValue(91);
		ctSetValueChars(2,3,pl);
		strcat(CatCmdT,"000");  // out level unused
		ctSendAnswer();
  }
}


void ctPM()
{
  ctDummy();
}






void ctPR()
{
	if (KwdGetSet == 'S') {
		if (Mode < 2) {
			if (CatCmdR[2] == '1') SWspeechcomp = 1; else SWspeechcomp = 0;
			dspSendCommand(9, SWspeechcomp);
		} else {
			if (CatCmdR[2] == '1') SWspot = 1; else SWspot = 0;
			dspSendCommand(9, SWspot);
		}
		zscShowSwitches();
	} else {
		if (Mode < 2) {
			if (SWspeechcomp == 1) CatCmdT[2] = '1'; else CatCmdT[2] = '0';
		} else {
			if (SWspot == 1) CatCmdT[2] = '1'; else CatCmdT[2] = '0';
		}
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
}


void ctPS()
{
  ctDummy();
}



void ctQC()
{
  ctDummy();
}






void ctQI()
{
  ctDummy();
}


void ctQR()
{
	uint16_t m;
	if(KwdGetSet=='S')
	{
		m = ctGetValueFromChars(3,1);
		if((m>=1)&&(m<=6))
		{
			MemSlotNo = m - 1;
			ddsLoadMemSlot(VFO,MemStackNo, MemSlotNo);   
  	  ddsApplyVfo(VFO);
			UpdateDDS = 1;
		}
		else
		{
			if(m==0){m=10;}
			MemStackNo = m-7;
			MemSlotNo = 255;
		}
		zscShowMemStackSlotNo();
	}
	else
	{
		m = MemSlotNo + 1;   // only reads back the slot number
		ctSetValueChars(2,2,m+10);
		ctSendAnswer();
	}

}



void ctRA()
{
	uint8_t ra;
	if(KwdGetSet=='S')
	{
	  if(ctGetValueFromChars(2,2) > 0){ra=1;}else{ra=0;};
		if(RFatten!=ra)
		{
		  RFatten = ra;
			scSetAtten(RFatten);
			zscShowSwitches();
		}
	}
	else
	{
    ctSetValueChars(2,2,RFatten);
		ctSendAnswer();
	}
}


void ctRC()
{
	if(KwdGetSet=='S')
	{
		cmCopyVfo();
	}
}

void ctRD()
{
  ctDummy();
}


void ctRG()
{
	uint16_t depth;
	if(KwdGetSet=='S')
	{
		if(Mode<2){return;};
		depth = (100*ctGetValueFromChars(2,3))/255;
		if(depth>100){depth=100;};
		dspChangeParamValue(61,depth/10);   
	}
	else
	{
		depth = (255*dspGetParamValue(61))/100;
		ctSetValueChars(2,3,depth);
		ctSendAnswer();
	}
}




void ctRL()
{
	uint16_t nr;
	if(KwdGetSet=='S')
	{
		nr = ctGetValueFromChars(2,2);   
		if(nr>12){nr=12;};
		if(nr==0){nr=1;};
		dspChangeParamValue(11,nr);   
	}
	else
	{
		nr = dspGetParamValue(11);
		ctSetValueChars(2,2,nr);
		ctSendAnswer();
  }
}


void ctRM()
{
  ctDummy();
}



void ctRU()
{  				
  ctDummy();
}


void ctRX()
{
  ctDummy();
}



void ctSA()
{
  ctDummy();
}


void ctSB()
{
	ctDummy();
}


void ctSC()
{
  uint16_t sc;	
	if(KwdGetSet=='S')
	{	
		sc = ctGetValueFromChars(2,1);  
		if((sc>=4)&&(sc<=6));
		if(ParamColour!=(sc-3))
		{
			ParamColour = sc-3;
			dspSetParameterColour();
			zscShowFilter();
		}	   
	}
	else
	{
		sc = ParamColour + 3;
		ctSetValueChars(2,1,sc);
		ctSendAnswer();
  }  
}






void ctSD()
{
	uint16_t sd;	
	if(KwdGetSet=='S')
	{
		if(Mode>1)
		{
			sd = ctGetValueFromChars(2,4)/5;   // Star - steps of 5mS
			if(sd>127){sd=127;};
			dspChangeParamValue(74,sd);
		}	   
	}
	else
	{
		if(Mode>1)
		{
		  sd = dspGetParamValue(74);
		  ctSetValueChars(2,4,((sd+15)/30)*150);
		}
		else
		{ctSetValueChars(2,4,0);}
		ctSendAnswer();
  }
}



void ctSH()
{
  ctDummy();
}


void ctSI()
{
  ctDummy();
}




void ctSL()
{
  ctDummy();
}



void ctSM()
{
	if(KwdGetSet=='G')
	{
		uint16_t n = SmeterdB;
    uint8_t i;
		n = (n*150)/540;
		CatCmdT[2]=CatCmdR[2];
		CatCmdT[3] = 0;
		if(CatCmdT[2]=='0')  // Smeter level main transceiver
		{
			for(i=0;i<4;i++)
			{
				CatCmdT[6-i] = (n%10) + 0x30;;
				n /= 10;
    	}
 		  CatCmdT[7] = 0;
		}
		else
		{strcat(CatCmdT,PMS(s_0000));};
		ctSendAnswer();
	}
}



void ctSQ()
{
	uint16_t squelch;
	if(KwdGetSet=='S')
	{
		if(CatCmdR[2]=='0')    // main transceiver
		{
			squelch = ctGetValueFromChars(3,3)/20;
			if(squelch>12){squelch=12;};
			dspChangeParamValue(22,squelch);
		}	 
	}
	else
	{
		squelch = 20*dspGetParamValue(22);
		if(CatCmdR[2]!='0'){squelch=0;};    // main transceiver
		CatCmdT[2]=CatCmdR[2];   //send P1 back
		CatCmdT[3]=0;
		ctSetValueChars(3,3,squelch);
		ctSendAnswer();
	}

}



void ctSR()
{
  ctDummy();
}


void ctSS()
{
  ctDummy();
}


void ctST()
{
  ctDummy();
}


void ctSU()
{
  ctDummy();
}


void ctSV()
{
  ctDummy();
}


void ctTC()
{
  ctDummy();
}


void ctTD()
{
  ctDummy();
}


void ctTI()
{
  ctDummy();
}


void ctTN()
{
  ctDummy();
}


void ctTO()
{
  ctDummy();
}


void ctTS()
{
  ctDummy();
}





void ctTX()
{
	if(KwdGetSet=='S')
	{
		if(CatCmdR[2]=='1')
		{portMOX |= (1 << MOX);}
		else
		{portMOX &= ~(1 << MOX);};
	}
}


void ctTY()
{
  ctDummy();
}


void ctUL()
{
  ctDummy();
}





void ctUP()   // for VFO A here read 'Active VFO' becuase no VFO B option
{
}

void ctVD()
{
	uint16_t vd;
	
	if(KwdGetSet=='S')
	{
		if(Mode<2)
		{
			vd = ctGetValueFromChars(2,4)/5;   // Star - steps of 5mS
			if(vd>127){vd=127;};
			dspChangeParamValue(74,vd);
		}	   
	}
	else
	{
		vd = dspGetParamValue(74);
		if(Mode<2)
		{ctSetValueChars(2,4,((vd+15)/30)*150);}
		else
		{ctSetValueChars(2,4,0);}
		ctSendAnswer();
  }
}


void ctVG()
{
	uint16_t vg;
	if(KwdGetSet=='S')
	{
		if(Mode>1){return;};
		vg = 10*ctGetValueFromChars(2,3);
		if(vg>90){vg=90;};
		dspChangeParamValue(72,vg);   
	}
	else
	{
		vg = dspGetParamValue(72);
		ctSetValueChars(2,3,vg/10);
		ctSendAnswer();
	}
}


void ctVR()
{
  ctDummy();
}




void ctVX()
{
	if(KwdGetSet=='S')
	{
		if (Mode < 2) {
			if (CatCmdR[2] == '1') SWvox = 1; else SWvox = 0;
			dspSendCommand(7, SWvox);   
		} else {
			if (CatCmdR[2] == '1') SWqsk = 1; else SWqsk = 0;
			dspSendCommand(7, SWqsk);   
		}
		zscShowSwitches();
	}
	else
	{
		if (Mode < 2) {
			if (SWvox == 1) CatCmdT[2]='1'; else CatCmdT[2] = '0'; 
		} else {
			if (SWqsk == 1) CatCmdT[2]='1'; else CatCmdT[2] = '0'; 
		}
		CatCmdT[3] = 0;
		ctSendAnswer();
	}
}



void ctXT()  		// dont use this to toggle XIT  - so no action if XIT on
{          	  // do not use this to toggle split off - so no action if Split ON
	if(KwdGetSet=='S')
	{
		if(CatCmdR[2]=='1'){cmXIT();}else{cmCopyVfo();};    
	}
	else
	{
		if(XIT==1){CatCmdT[2]='1';}else{CatCmdT[2]='0';};
		CatCmdT[3]=0;
		ctSendAnswer();
	}

}


